package ru.yandex.chemodan.uploader.registry;

import java.net.URI;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.mulca.MulcaUploadManager;
import ru.yandex.chemodan.test.ReflectionUtils;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.config.UploaderCoreContextConfigurationForTests;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.mulca.SimultaneousMulcaUploadManager;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.processors.UploadToDefaultProcessor;
import ru.yandex.chemodan.uploader.registry.record.Digests;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.DigestCalculationStatus;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.test.TestMpfsRequestRecordCreator;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.registry.CallbackResponse;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.RequestMeta;
import ru.yandex.commune.uploader.registry.StageListenerPool;
import ru.yandex.commune.uploader.registry.StageProperties;
import ru.yandex.commune.uploader.registry.StageResult;
import ru.yandex.commune.uploader.registry.StageSemaphores;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareCalculateDigestFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareCheckWithAntivirusFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareCommitUploadFileProgressFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareContentInfoFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareExtractExifInfoFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareGenerateOnePreviewForImageFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareMockForAnyRecordUpdate;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.preparePreviewUploadFileToMulcaFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareUploadDigestToMulcaFMock;
import static ru.yandex.chemodan.uploader.registry.ChemodanRequestDirectorWithMockedUploadRegistryTestUtils.prepareUploadFileToMulcaFMock;
import static ru.yandex.chemodan.uploader.test.DomainUtils.mid;
import static ru.yandex.chemodan.uploader.test.DomainUtils.uploadInfo;

/**
 * @author nshmakov
 * @author qwwdfsad
 */
@ContextConfiguration(classes = UploaderCoreContextConfigurationForTests.class)
public class UploadToDefaultTest extends AbstractTest {
    private final TestMpfsRequestRecordCreator recordCreator = new TestMpfsRequestRecordCreator();

    @Mock
    private UploadRegistry<MpfsRequestRecord> uploadRegistryMock;
    @Mock
    private Stages stagesMock;
    @Mock
    DocviewerClient docviewerClient;
    @Captor
    private ArgumentCaptor<Function2<RequestMeta, HostInstant, MpfsRequestRecord.MarkMulcaIdsForRemove>> consCaptor;

    // Dependencies
    @Autowired
    MulcaUploadManager mulcaUploadManager;
    @Autowired
    private StageListenerPool<Record<?>> stageListenerPool;
    @Autowired
    private StageProperties stageProperties;
    @Autowired
    private StageSemaphores stageSemaphores;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    @Qualifier("experimental")
    private BleedingEdge experimentalBleedingEdge;
    @Autowired
    private SimultaneousMulcaUploadManager simultaneousMulcaUploadManager;

    private RequestStatesHandler requestStatesHandler;

    // Test subject
    UploadToDefaultProcessor uploadToDefaultProcessor;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        requestStatesHandler = new RequestStatesHandler(
                stageListenerPool, stageProperties, stageSemaphores, uploadRegistryMock, stagesMock);
        uploadToDefaultProcessor = new UploadToDefaultProcessor(requestStatesHandler, stagesMock,
                docviewerClient, mulcaUploadManager, localFileManager, experimentalBleedingEdge,
                simultaneousMulcaUploadManager);
    }

    @Test
    public void uploadToDefault() {
        String filePath = "/file";
        UidOrSpecial uid = UidOrSpecial.uid(PassportUid.cons(100L));
        IncomingFile incomingFile = new IncomingFile(Option.empty(), Option.empty(), new File2("file"));
        File2 previewFile = new File2("preview");
        MpfsRequestRecord.UploadToDefault record = createRecord(filePath, uid, incomingFile, Option.empty());
        final Option<String> contentTypeO = Option.of("image/jpeg");

        prepareMockForAnyRecordUpdate(uploadRegistryMock, record);
        prepareContentInfoFMock(stagesMock, incomingFile, Option.of(filePath), contentTypeO);
        prepareCommitUploadFileProgressFMock(stagesMock, record);
        prepareUploadFileToMulcaFMock(stagesMock, incomingFile, uid);
        prepareCalculateDigestFMock(stagesMock, record.meta.id, incomingFile);
        prepareUploadDigestToMulcaFMock(stagesMock, incomingFile, uid);
        prepareExtractExifInfoFMock(stagesMock, incomingFile, record, contentTypeO.get());
        prepareGenerateOnePreviewForImageFMock(stagesMock, incomingFile, record, previewFile, contentTypeO);
        preparePreviewUploadFileToMulcaFMock(stagesMock, previewFile, uid);
        prepareCheckWithAntivirusFMock(stagesMock, incomingFile);
        prepareMocksForExif();

        uploadToDefaultProcessor.process(record);
    }

    @Test
    public void shouldRollbackIfSecondCommitResponseIs409() {
        MulcaId fileMid = mid(1);
        MulcaId digestMid = mid(2);
        MpfsRequestRecord.UploadToDefault record = createRecordWithFinishedProcessingStages(
                fileMid, digestMid, Option.empty());

        prepareMocksForProcessingStages(record);
        uploadToDefaultProcessor.process(record);

        assertMidsWasCaptured(Cf.list(fileMid, digestMid));
    }

    @Test
    public void shouldRollbackIfFinalCommitResponseIs409() {
        MulcaId fileMid = mid(1);
        MulcaId digestMid = mid(2);
        MulcaId previewMid = mid(3);
        MpfsRequestRecord.UploadToDefault record =
                createRecordWithFinishedPostProcessingStages(fileMid, digestMid, previewMid, Option.of(new File2("digest")));

        prepareMocksForPostProcessingStages(record);
        prepareMocksForExif();
        uploadToDefaultProcessor.process(record);

        assertMidsWasCaptured(Cf.list(fileMid, digestMid, previewMid));
    }

    private void assertMidsWasCaptured(ListF<MulcaId> expected) {
        verify(uploadRegistryMock).saveRecord(consCaptor.capture());
        MpfsRequestRecord.MarkMulcaIdsForRemove value = consCaptor.getValue()
                .apply(new RequestMeta(UploadRequestId.valueOf("1"), Instant.now()), HostInstant.hereAndNow());
        ListF<MulcaId> actual = value.getRequest().mulcaIds;
        Assert.equals(actual, expected);
    }

    private void prepareMocksForPostProcessingStages(MpfsRequestRecord.UploadToDefault record) {
        prepareMocksForProcessingStages(record);
        Function0 functionMock = mock(Function0.class);
        when(functionMock.asFunction()).thenReturn(mock(Function.class));
        when(stagesMock.extractExifInfoF(
                    any(UploadRequestId.class), any(InputStreamSource.class), any(Option.class), any(), any()))
                .thenReturn(functionMock);
        when(stagesMock.generateOnePreviewForImageF(any(UploadRequestId.class), any(InputStreamSource.class),
                any(Option.class), any(PreviewSizeParameter.class), anyBoolean(), anyBoolean(), any(Option.class)))
                    .thenReturn(functionMock);
        when(stagesMock.checkWithAntivirusF(any(Option.class), any(InputStreamSource.class))).thenReturn(functionMock);
    }

    private void prepareMocksForProcessingStages(MpfsRequestRecord.UploadToDefault record) {
        Function0 functionMock = mock(Function0.class);
        when(functionMock.asFunction()).thenReturn(mock(Function.class));
        prepareMockForAnyRecordUpdate(uploadRegistryMock, record);
        when(stagesMock.contentInfoF(any(IncomingFile.class), any(Option.class))).thenReturn(functionMock);
        when(stagesMock.uploadFileToMulcaF(any(InputStreamSource.class), any(UidOrSpecial.class)))
                .thenReturn(functionMock);
        when(stagesMock.calculateDigestF(any(UploadRequestId.class), any(InputStreamSource.class)))
                .thenReturn(functionMock);
        when(stagesMock.uploadDigestToMulcaF(any(File2.class), any(UidOrSpecial.class)))
                .thenReturn(functionMock);
        when(stagesMock.commitUploadFileProgressF(eq(record), any(String.class))).thenReturn(
                () -> StageResult.prematureSuccess(callbackResponseOption(409)));
    }

    private void prepareMocksForExif() {
        when(stagesMock.extractFullExifInJsonF(any(File2.class))).thenReturn(Function0.constF("exif"));
    }

    private MpfsRequestRecord.UploadToDefault createRecord(String filePath, UidOrSpecial uidOrSpecial,
            IncomingFile incomingFile, Option<URI> callbackUriO)
    {
        MpfsRequestRecord.UploadToDefault record = recordCreator.generateRandomUploadToDefaultRecord();

        ReflectionUtils.setFieldNewValue("chemodanFile", record.getRequest(),
                ChemodanFile.consWithoutFileId(uidOrSpecial, filePath));
        ReflectionUtils.setFieldNewValue("callbackUri", record.getRequest(), callbackUriO);

        record.getStatus().userFile.complete(incomingFile);
        return record;
    }

    private MpfsRequestRecord.UploadToDefault createRecordWithFinishedProcessingStages(
            MulcaId fileMid, MulcaId digestMid, Option<File2> digestFile)
    {
        IncomingFile file = new IncomingFile(Option.empty(), Option.empty(), new File2("file"));
        MpfsRequestRecord.UploadToDefault record =
                createRecord("/path", UidOrSpecial.uid(PassportUid.cons(1L)), file,
                        Option.of(URI.create("localhost")));
        record.getStatus().payloadInfo.complete(new ContentInfo(Option.of("image/jpeg"), 100L, "md5", "sha256"));
        record.getStatus().postProcess.commitFileInfo.complete(callbackResponseOption(200));
        record.getStatus().postProcess.fileMulcaUploadInfo.complete(uploadInfo(fileMid));
        record.getStatus().postProcess.digestCalculationStatus.complete(digestCalculationStatus(digestFile));
        record.getStatus().postProcess.digestMulcaUploadInfo.complete(uploadInfo(digestMid));
        record.getStatus().userFileOnTheFlyDigests.set(State.<Digests>initial().skipSuccess());
        return record;
    }

    private MpfsRequestRecord.UploadToDefault createRecordWithFinishedPostProcessingStages(MulcaId fileMid,
            MulcaId digestMid, MulcaId previewId, Option<File2> digestFile)
    {
        MpfsRequestRecord.UploadToDefault record = createRecordWithFinishedProcessingStages(fileMid, digestMid, digestFile);
        record.getStatus().postProcess.commitFileUpload.complete(callbackResponseOption(200));
        record.getStatus().postProcess.exifInfo.complete(new ExifInfo(Option.empty(), Option.empty()));
        record.getStatus().postProcess.previewImageStatus.generateOnePreview.complete(new GenerateImageOnePreviewResult(
                new File2("preview"), ImageFormat.JPEG, Option.empty(), Dimension.valueOf(100, 100), Option.empty()));
        record.getStatus().postProcess.previewImageStatus.previewMulcaUploadInfo.complete(uploadInfo(previewId));
        record.getStatus().postProcess.antivirusResult2.complete(AntivirusResult.HEALTHY);
        return record;
    }

    private CallbackResponseOption callbackResponseOption(int status) {
        return new CallbackResponseOption(Option.of(new CallbackResponse(status, "", "", Cf.map(), Option.empty())));
    }

    private DigestCalculationStatus digestCalculationStatus(Option<File2> digestFile) {
        return new DigestCalculationStatus(digestFile, Option.of(1L));
    }
}
