package ru.yandex.chemodan.uploader.web;

import org.apache.http.client.methods.HttpGet;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.test.ClassLoaderTestUtils;
import ru.yandex.chemodan.test.MulcaTestUtils;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.av.BaseAntivirus;
import ru.yandex.chemodan.uploader.exif.ExifToolTest;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
@ContextConfiguration(classes = {
        WebRecalcDigestsTest.Config.class
})
public class WebRecalcDigestsTest extends AbstractWebTestSupport {

    @Configuration
    public static class Config {

        @Bean
        public BaseAntivirus antivirus() {
            return new BaseAntivirus(true, DataSize.MEGABYTE, true) {
                @Override
                public boolean isEnabled() {
                    return true;
                }

                @Override
                protected AntivirusResult doCheck(InputStreamSource file) {
                    return AntivirusResult.INFECTED;
                }
            };
        }

    }

    @Autowired
    private MulcaClient mulcaClient;

    @Test
    public void recalcDigests() {
        recalcDigests(true, true, true);
    }

    @Test
    public void recalcDigestsWithoutWebdavDigest() {
        recalcDigests(false, false, false);
    }

    @Test
    public void recaclDigestsWithNotFoundError() {
        try {
            recalcDigests(Option.empty(), true, true, true);
            Assert.fail();
        } catch (HttpException e) {
            Assert.equals(HttpStatus.SC_404_NOT_FOUND, e.getStatusCode().get());
        }
    }

    private void recalcDigests(boolean withWebdav, boolean withExif, boolean withVirusCheck) {
        String json = recalcDigests(ClassLoaderTestUtils.byteStreamSourceForResource(ExifToolTest.class, "small.jpg"), withWebdav, withExif, withVirusCheck);
        JsonObject response = (JsonObject) JsonParser.getInstance().parse(json);

        Assert.equals("e5878b2b21194e64519d2b61c98d9adc", ((JsonString) response.get("md5")).getString());
        Assert.equals("7c44a201e6ab2d2998812a3cacdd2dc60e60107f4b0dbf65fa451b45d88d3a69",
                ((JsonString) response.get("sha256")).getString());
        Assert.equals(19233L, ((JsonNumber) response.get("content-length")).longValue());
        Assert.equals("image/jpeg", ((JsonString) response.get("content-type")).getString());
        assertOptionExistence(response.getO("webdav-digest-mulca-id"), withWebdav);
        assertOptionExistence(response.getO("extracted-exif-data"), withExif);
        assertOptionExistence(response.getO("antivirus-results"), withVirusCheck);
    }

    private static void assertOptionExistence(Option<?> option, boolean expected) {
        if (expected) {
            Assert.some(option);
        } else {
            Assert.none(option);
        }
    }

    private String recalcDigests(InputStreamSource source, boolean withWebdav, boolean withExif, boolean withVirusCheck) {
        return recalcDigests(Option.of(source), withWebdav, withExif, withVirusCheck);
    }

    private String recalcDigests(Option<InputStreamSource> source, boolean withWebdav, boolean withExif, boolean withVirusCheck) {
        MulcaId originalMulcaId = source.isPresent() ?
                mulcaClient.upload(source.get(), "tmp") :
                MulcaTestUtils.FAKE_MULCA_ID;

        String url = UrlUtils.addParameters(
                "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.RECALC_DIGESTS_URL,
                Cf.<String, Object>map(ApiArgs.MULCA_ID, originalMulcaId.getStidCheckNoPart()));
        if (withWebdav) {
            url = UrlUtils.addParameter(url, ApiArgs.WITH_WEBDAV, true);
        }
        if (withExif) {
            url = UrlUtils.addParameter(url, ApiArgs.WITH_EXIF, true);
        }
        if (withVirusCheck) {
            url = UrlUtils.addParameter(url, ApiArgs.WITH_VIRUS_CHECK, true);
        }
        return ApacheHttpClientUtils.executeReadString(new HttpGet(url));
    }

}
