package ru.yandex.chemodan.uploader.web;

import org.apache.http.client.methods.HttpGet;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.test.MulcaTestManager;
import ru.yandex.chemodan.test.MulcaTestUtils;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.av.AntivirusTestUtils;
import ru.yandex.chemodan.uploader.av.BaseAntivirus;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

/**
 * @author akirakozov
 */
@ContextConfiguration(classes = {WebRecheckVirusesTest.Config.class})
public class WebRecheckVirusesTest extends AbstractWebTestSupport {
    @Configuration
    public static class Config {
        @Bean
        public BaseAntivirus antivirus() {
            return antivirus;
        }
    }

    @Autowired
    private MulcaClient mulcaClient;
    @Mock
    private static BaseAntivirus antivirus;

    @BeforeClass
    public static void init() {
        antivirus = Mockito.mock(BaseAntivirus.class);
        when(antivirus.isEnabled()).thenReturn(true);
    }

    private String recheckViruses(InputStreamSource source) {
        return recheckViruses(Option.of(source));
    }

    private String recheckViruses(Option<InputStreamSource> source) {
        if (source.isPresent()) {
            MulcaTestManager testManager = new MulcaTestManager(mulcaClient);
            return testManager.withUploadedToMulcaFile(source.get(), false, (Function<MulcaId, String>) this::recheckViruses);
        } else {
            return recheckViruses(MulcaTestUtils.FAKE_MULCA_ID);
        }
    }

    private String recheckViruses(MulcaId originalMulcaId) {
        String url = UrlUtils.addParameters(
                "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.RECHECK_VIRUSES_URL,
                Cf.map(ApiArgs.MULCA_ID, originalMulcaId.getStidCheckNoPart()));

        return ApacheHttpClientUtils.executeReadString(new HttpGet(url));
    }

    @Test
    public void recheckVirusesHealthy() {
        when(antivirus.check(any())).thenReturn(AntivirusResult.HEALTHY);

        String json = recheckViruses(AntivirusTestUtils.healthyInputStreamSource());
        JsonObject response = (JsonObject) JsonParser.getInstance().parse(json);
        Assert.equals("healthy", ((JsonString) response.get("result")).getString());
    }

    @Test
    public void recheckVirusesInfected() {
        when(antivirus.check(any())).thenReturn(AntivirusResult.INFECTED);

        String json = recheckViruses(AntivirusTestUtils.infectedInputStreamSource());
        JsonObject response = (JsonObject) JsonParser.getInstance().parse(json);
        Assert.equals("infected", ((JsonString) response.get("result")).getString());
    }

    @Test
    public void recheckVirusesWithNotFoundError() {
        try {
            recheckViruses(Option.empty());
            Assert.fail();
        } catch (HttpException e) {
            Assert.equals(HttpStatus.SC_404_NOT_FOUND, e.getStatusCode().get());
        }
    }
}
