package ru.yandex.chemodan.uploader.web;

import java.io.StringReader;

import org.apache.http.client.methods.HttpGet;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.test.TestManager;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author akirakozov
 */
@YaExternal
public class WebRegeneratePreviewTest extends AbstractWebTestSupport {

    @Autowired
    private MulcaClient mulcaClient;

    private void regeneratePreview(
            final InputStreamSource source,
            final String contentType,
            final Option<String> fileName)
    {
        TestManager.withEmptyTemporaryFile("regenerate", tmpFile -> {
            TestManager.copyISStoFile2(source, tmpFile);
            MulcaId originalMulcaId = mulcaClient.upload(tmpFile, "tmp");

            String url = UrlUtils.addParameters(
                    "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.REGENERATE_PREVIEW_URL,
                    Cf.<String, Object>map(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                        .plus1(ApiArgs.MULCA_ID, originalMulcaId.getStidCheckNoPart())
                        .plus1(ApiArgs.CONTENT_TYPE, contentType)
                        .plus1(ApiArgs.FILE_SIZE, tmpFile.length())
                        .plus1(ApiArgs.FILE_NAME, fileName.getOrElse("")));

            String xml = ApacheHttpClientUtils.executeReadString(new HttpGet(url));
            String previewMulcaId =
                    Dom4jUtils.readRootElement(new StringReader(xml)).element("mulca-id").getText();
            Assert.isTrue(StringUtils.isNotEmpty(previewMulcaId));
        });
    }

    @Test
    public void regenerateImagePreview() {
        regeneratePreview(
                new ClassPathResourceInputStreamSource(WebRegeneratePreviewTest.class, "300.jpg"),
                "image/jpg", Option.empty());
    }

    @Test
    public void regenerateDjvuPreview() {
        regeneratePreview(
                new ClassPathResourceInputStreamSource(WebRegeneratePreviewTest.class, "bread.djvu"),
                "application/djvu", Option.empty());
    }

    @Test
    public void regenerateBrokenImagePreview() {
        try {
            regeneratePreview(
                    new ByteArrayInputStreamSource("It's not image".getBytes()),
                    "image/jpg", Option.empty());
            Assert.fail();
        } catch (HttpException e) {
            Assert.equals(HttpStatus.SC_400_BAD_REQUEST, e.getStatusCode().get());
        }
    }

    @Test
    public void regenerateNotSupportedMimeType() {
        try {
            regeneratePreview(new ByteArrayInputStreamSource(new byte[]{1, 2}), "audio/mp3", Option.empty());
            Assert.fail();
        } catch (HttpException e) {
            Assert.equals(HttpStatus.SC_400_BAD_REQUEST, e.getStatusCode().get());
        }
    }

    @Test
    public void regenerateFileNotFound() {
        try {
            String url = UrlUtils.addParameters(
                    "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.REGENERATE_PREVIEW_URL,
                    Cf.<String, Object>map(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                        .plus1(ApiArgs.MULCA_ID, "fake-mulca-id")
                        .plus1(ApiArgs.CONTENT_TYPE, "image/jpg")
                        .plus1(ApiArgs.FILE_SIZE, 1));

            ApacheHttpClientUtils.executeReadString(new HttpGet(url));
            Assert.fail();
        } catch (HttpException e) {
            Assert.equals(HttpStatus.SC_404_NOT_FOUND, e.getStatusCode().get());
        }
    }

    @Test
    public void regenerateDocumentPreview() {
        regeneratePreview(
                new ClassPathResourceInputStreamSource(DocviewerClient.class, "test.pdf"),
                "application/pdf", Option.empty());
    }

    @Test
    public void regenerateDocumentPreviewWithUnknownMimeType() {
        regeneratePreview(
                new ClassPathResourceInputStreamSource(DocviewerClient.class, "test.pdf"),
                "application/octet-stream", Option.of("test.pdf"));
    }

    @Test
    public void invalidMp4() {
        try {
            regeneratePreview(
                    new ClassPathResourceInputStreamSource(WebRegeneratePreviewTest.class, "invalid.mp4"),
                    "video/mp4", Option.of("invalid.mp4")
            );
            Assert.fail();
        } catch (HttpException ex) {
            Assert.isTrue(ex.statusCodeIs(HttpStatus.SC_400_BAD_REQUEST));
        }
    }
}
