package ru.yandex.chemodan.uploader.web;

import org.apache.http.client.HttpClient;
import org.dom4j.Element;
import org.glassfish.grizzly.http.Method;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.ping.HostsProvider;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.browser.BrowserMdsTestContextConfiguration;
import ru.yandex.chemodan.uploader.exif.ExifTool;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo.GeoCoords;
import ru.yandex.chemodan.uploader.services.ServiceFileId;
import ru.yandex.chemodan.uploader.web.client.UploaderClient;
import ru.yandex.chemodan.util.http.HttpClientUtils;
import ru.yandex.chemodan.util.test.StubServerUtils;
import ru.yandex.chemodan.util.test.TestUser;
import ru.yandex.commune.uploader.registry.UploadRequestStatus;
import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.inside.mds.Mds;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.stringContent;
import static com.xebialabs.restito.semantics.Condition.composite;
import static com.xebialabs.restito.semantics.Condition.method;
import static com.xebialabs.restito.semantics.Condition.parameter;
import static com.xebialabs.restito.semantics.Condition.uri;

/**
 * @author vavinov
 * @author ssytnik
 */
@YaExternal
@ContextConfiguration(classes = {BrowserMdsTestContextConfiguration.class})
public class WebUploadFromServiceTest extends AbstractWebTestSupport {

    private static final String NON_EXISTING_VK_IMAGE_URL = "http://retrydisabletrue";
    private static final String VK_IMAGE_URL_WITH_REDIRECT = "http://goo.gl/L7Yplr";

    @Autowired
    private HostsProvider hostsProvider;
    @Value("${hancom.host-port}")
    private HostPort hancomHostPort;
    @Autowired
    private Mds browserMds;

    @Before
    public void init() {
        hostsProvider.start();
    }

    @After
    public void destroy() {
        hostsProvider.stop();
    }

    @Test
    public void uploadFromMail2ToDisk() {
        ServiceFileId serviceFileId = ServiceFileId.valueOf("4006336590:161848111608628001/1.1");

        doUploadFromService(
                ChemodanService.MAIL2,
                serviceFileId,
                a -> Assert.equals(11966, a.readBytes().length),
                Option.empty());
    }

    @Test
    public void uploadFromMdsForYaBrowser() {
        String data = "my data";
        Option<MdsFileKey> mdsFileKey = Option.empty();
        try {
            String filename = Random2.R.nextString(10);
            MdsPostResponse response = browserMds.upload(filename, new ByteArrayInputStreamSource(data.getBytes()));
            mdsFileKey = Option.of(response.getKey());
            ServiceFileId serviceFileId = new ServiceFileId(TestUser.uid, mdsFileKey.get().serialize());

            doUploadFromService(
                    ChemodanService.BROWSER,
                    serviceFileId,
                    a -> Assert.equals(data, a.readText()),
                    Option.empty());
        } finally {
            mdsFileKey.forEach(browserMds::delete);
        }
    }

    @Test
    public void uploadFileFromHancom() {
        String data = "my data content";
        StubServerUtils.withStubServer(
                hancomHostPort.getPort(),
                s -> {
                    whenHttp(s).match(
                        composite(
                                method(Method.GET),
                                uri("/weboffice/api/v1/exportDocument.do"),
                                parameter("docId", "documentId-123"))
                        ).then(stringContent(data));

                    doUploadFromService(
                            ChemodanService.HANCOM,
                            new ServiceFileId(TestUser.uid, "weboffice/documentId-123"),
                            a -> Assert.equals(data, a.readText()),
                            Option.empty());
                }
         );
    }

    @Test
    public void uploadFromVkToDisk() {
        Long timestampInSec = 1346504877L;
        Instant ts = new Instant(timestampInSec * 1000);
        GeoCoords coords = new GeoCoords(42.2, 18.9);
        doUploadFromService(ChemodanService.VKONTAKTE,
                new ServiceFileId(TestUser.uid, "123"),
                checkExifF(coords, ts),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of(timestampInSec),
                Option.of("http://cs616119.vk.me/v616119352/2eed/sEwR8Le8YCA.jpg"),
                Option.of(coords.getLatitude()),
                Option.of(coords.getLongitude()),
                UploadRequestStatus.Result.COMPLETED);
    }

    @Test
    public void uploadFromGooglePlusToDisk() {
        doUploadFromService(ChemodanService.GOOGLE,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of(1346504877L),
                Option.of("https://lh3.googleusercontent.com/cu_kspTOP8JCORcpSYfOwvm8SNwZmsAQfKvnMnujnOw=w1077-h808-no"),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.COMPLETED);
    }

    @Test
    public void uploadFromInstagramToDisk() {
        doUploadFromService(ChemodanService.INSTAGRAM,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of(1346504877L),
                Option.of("http://scontent-a.cdninstagram.com/hphotos-xaf1/t50.2886-16/10647970_1479116089012464_320079811_n.mp4"),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.COMPLETED);
    }

    @Test
    public void uploadGifImageFromMailru() {
        doUploadFromService(ChemodanService.MAILRU,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of(1346504877L),
                Option.of("http://content-9.foto.my.mail.ru/mail/lara241261/_guestbook/h-2932.gif"),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.COMPLETED);
    }

    @Test
    public void uploadNonExistingImageWithDisableRetriesFromVkToDisk() {
        doUploadFromService(ChemodanService.VKONTAKTE,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.of(true),
                Option.of(false),
                Option.of(1346504877L),
                Option.of(NON_EXISTING_VK_IMAGE_URL),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.FAILED);
    }

    @Test
    public void uploadImageWithDisableRetriesFromForbiddenHostToDisk() {
        doUploadFromService(ChemodanService.WEB,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.of(true),
                Option.of(false),
                Option.of(1346504877L),
                Option.of("http://mpfs.disk.yandex.net/1.jpg"),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.FAILED);
    }

    @Test
    public void uploadImageWithDisableRedirectsFromVkViaShortLinkToDisk() {
        doUploadFromService(ChemodanService.VKONTAKTE,
                new ServiceFileId(TestUser.uid, "123"),
                Function1V.nop(),
                Option.empty(),
                Option.of(false),
                Option.of(true),
                Option.of(1346504877L),
                Option.of(VK_IMAGE_URL_WITH_REDIRECT),
                Option.empty(),
                Option.empty(),
                UploadRequestStatus.Result.FAILED);
    }

    private void doUploadFromService(ChemodanService service,
            ServiceFileId serviceFileId,
            Function1V<InputStreamSource> checkResult,
            Option<Integer> maxFileSize,
            Option<Boolean> disableRetries,
            Option<Boolean> disableRedirects,
            Option<Long> createdSeconds,
            Option<String> serviceFileUrl,
            Option<Double> latitude,
            Option<Double> longitude,
            UploadRequestStatus.Result expectedResult)
    {
        HttpClient httpClient = ApacheHttpClientUtils.singleConnectionClient(getHttpClientTimeout());
        try {
            MapF<String, Object> params = UploaderClient.toParameterMap(CHE_FILE)
                    .plus1(ApiArgs.SOURCE_SERVICE, service)
                    .plus1(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                    .plus1(ApiArgs.SERVICE_FILE_ID, serviceFileId.toSerializedString());
            if (createdSeconds.isPresent()) {
                params = params.plus1(ApiArgs.CREATED, createdSeconds.get());
            }
            if (serviceFileUrl.isPresent()) {
                params = params.plus1(ApiArgs.SERVICE_FILE_URL, serviceFileUrl.get());
            }
            if (latitude.isPresent()) {
                params = params.plus1(ApiArgs.LATITUDE, latitude.get());
            }
            if (longitude.isPresent()) {
                params = params.plus1(ApiArgs.LONGITUDE, longitude.get());
            }
            if (maxFileSize.isPresent()) {
                params = params.plus1(ApiArgs.MAX_FILE_SIZE, 0);
            }
            if (disableRetries.isPresent()) {
                params = params.plus1(ApiArgs.DISABLE_RETRIES, disableRetries.get());
            }
            if (disableRedirects.isPresent()) {
                params = params.plus1(ApiArgs.DISABLE_REDIRECTS, disableRedirects.get());
            }

            Element urls = HttpClientUtils.parseXmlResponse(httpClient, HttpClientUtils.httpPost(
                    "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.UPLOAD_FROM_SERVICE + "/",
                    params));

            UploaderClient.UploadFromServiceStatusXml result = new UploaderClient.UploadFromServiceStatusXml(
                    pollStatusUntilFinished(httpClient, urls.attributeValue("poll-result")));

            Assert.equals(expectedResult.name().toLowerCase(), result.getStatus());
            if (expectedResult == UploadRequestStatus.Result.COMPLETED) {
                MulcaId mulcaId = result.getFileMulcaId().getOrThrow(result.getXml().asXML());
                try {
                    checkResult.apply(mulcaClient.download(mulcaId));
                } finally {
                    mulcaClient.delete(mulcaId);
                }
            }
        } finally {
            ApacheHttpClientUtils.stopQuietly(httpClient);
        }
    }

    private Function1V<InputStreamSource> checkExifF(final GeoCoords coords, final Instant ts) {
        return new Function1V<InputStreamSource>() {
            public void apply(final InputStreamSource a) {
                File2.withNewTempFile(new Function1V<File2>() {
                    public void apply(File2 file) {
                        a.readTo(file);
                        ExifInfo exif = ExifTool.INSTANCE.getExif(file);
                        Assert.equals(coords, exif.getGeoCoords().get());
                        Assert.equals(coords, exif.getGeoCoords().get());
                        // Parser parse datetime in default timezone CHEMODAN-16607
                        Assert.equals(
                                new LocalDateTime(ts, DateTimeZone.UTC),
                                new LocalDateTime(exif.getCreationDate().get(), DateTimeZone.getDefault()));
                    }
                }.asFunctionReturnNull());
            }
        };
    }

    private Function1V<InputStreamSource> checkMd5(final String md5) {
        return new Function1V<InputStreamSource>() {
            public void apply(InputStreamSource a) {
                Assert.equals(md5, Md5.A.digest(a).hex());
            }
        };
    }

    private void doUploadFromService(ChemodanService service, ServiceFileId serviceFileId,
            Function1V<InputStreamSource> checkResult, Option<Integer> maxFileSize)
    {
        UploadRequestStatus.Result expectedResult = maxFileSize.isPresent() ?
                UploadRequestStatus.Result.FAILED : UploadRequestStatus.Result.COMPLETED;
        doUploadFromService(service, serviceFileId, checkResult, maxFileSize,
                Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), expectedResult);
    }
}
