package ru.yandex.chemodan.uploader.web;

import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpHead;
import org.dom4j.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.exif.ExifTool;
import ru.yandex.chemodan.uploader.preview.PreviewImageManagerTest;
import ru.yandex.chemodan.uploader.util.UploaderHttpClientUtils;
import ru.yandex.chemodan.uploader.web.client.UploaderClient;
import ru.yandex.chemodan.util.http.HttpClientUtils;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.commune.uploader.registry.UploadRequestStatus;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.env.Environment;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.gzip.GzipUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author vavinov
 */
public class WebUploadTest extends AbstractWebTestSupport {
    private static final Logger logger = LoggerFactory.getLogger(WebUploadTest.class);

    private UploaderClient uploaderClient;
    private HttpClient httpClient;

    @Before
    public void beforeUploadTest() {
        uploaderClient = newUploaderClient();
        Timeout timeout = Environment.isDeveloperNotebook() ? Timeout.unlimited() : Timeout.seconds(5);
        httpClient = ApacheHttpClientUtils.singleConnectionClient(timeout);
    }

    @After
    public void afterUploadTest() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
        httpClient = null;
        uploaderClient.close();
        uploaderClient = null;
    }

    @Test
    public void uploadPostMultipartCheckStatusAndDownload() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.notEquals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), initialStatus.getStatus());
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        String testText = "hello, world!";

        int code = urls.uploadMultipart(InputStreamSourceUtils.bytes(testText.getBytes()));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());

        MulcaId mulcaFileId = afterUploadStatus.getFileMulcaId()
                .getOrThrow(afterUploadStatus.getXml().asXML());
        Assert.equals(testText, mulcaClient.download(mulcaFileId).readText());
        Assert.equals(Md5.A.digest(testText).hex(), afterUploadStatus.getIncomingFileMd5());
    }

    @Test
    public void uploadCheckStatusAndDownload() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        String testText = "hello, world!";

        int code = urls.upload(InputStreamSourceUtils.bytes(testText.getBytes()));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());

        MulcaId mulcaFileId = afterUploadStatus.getFileMulcaId()
                .getOrThrow(afterUploadStatus.getXml().asXML());
        Assert.equals(testText, mulcaClient.download(mulcaFileId).readText());
        Assert.equals(Md5.A.digest(testText).hex(), afterUploadStatus.getIncomingFileMd5());
    }

    @Test
    public void uploadWithGzipContentEncodingAndChunkedTransferEncoding() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        String testText = "hello, world!";
        byte[] encodedText = GzipUtils.zip(testText.getBytes());

        int code = HttpClientUtils.downloadStatus(httpClient,
                HttpClientUtils.httpPutZipped(urls.getPostUrl(), encodedText));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());

        MulcaId mulcaFileId = afterUploadStatus.getFileMulcaId()
                .getOrThrow(afterUploadStatus.getXml().asXML());
        Assert.equals(testText, mulcaClient.download(mulcaFileId).readText());
    }

    @Test
    public void uploadThenStopToResumeThenResume() throws Exception {
        String text = "0123456789";

        UploaderClient.UploadUrls urls = startPartialUpload(text);
        ThreadUtils.sleep(100);

        Assert.assertListsEqual(Cf.list("5"),
                HttpClientUtils.parseResponseHeaders(httpClient, new HttpHead(urls.getPostUrl()))
                        .getTs("Content-Length"));

        // resume after current end
        Assert.assertEquals(HttpStatus.SC_201_CREATED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutRanged(urls.getPostUrl(), text.substring(5, 10).getBytes(), 5, 9, 10)));
    }

    @Test
    public void uploadAndTryToResumeWithIncorrectRange() throws Exception {
        String text = "0123456789";

        UploaderClient.UploadUrls urls = startPartialUpload(text);
        ThreadUtils.sleep(100);

        // resume with incorrect range
        Assert.assertEquals(HttpStatus.SC_412_PRECONDITION_FAILED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutRanged(urls.getPostUrl(), text.substring(3, 10).getBytes(), 3, 9, 10)));

        // resume after current end with correct range
        Assert.assertEquals(HttpStatus.SC_201_CREATED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutRanged(urls.getPostUrl(), text.substring(5, 10).getBytes(), 5, 9, 10)));
    }

    @Test
    public void uploadInTwoRangedPuts() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        String text = "0123456789";

        Assert.assertEquals(HttpStatus.SC_404_NOT_FOUND,
                HttpClientUtils.downloadStatus(httpClient, new HttpHead(urls.getPostUrl())));

        Assert.equals(HttpStatus.SC_202_ACCEPTED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutRanged(urls.getPostUrl(),
                        text.substring(0, 8).getBytes(), 0, 7, text.length())));
        Assert.assertListsEqual(Cf.list("8"),
                HttpClientUtils.parseResponseHeaders(httpClient, new HttpHead(urls.getPostUrl()))
                        .getTs("Content-Length"));

        Assert.equals(HttpStatus.SC_201_CREATED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutRanged(urls.getPostUrl(),
                        text.substring(8).getBytes(), 8, 9, 10)));
    }

    @Test
    public void uploadInTwoRangedPutsZipped() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        String text = "0123456789";
        byte[] zippedPart1 = GzipUtils.zip(text.substring(0, 8).getBytes());
        byte[] zippedPart2 = GzipUtils.zip(text.substring(8).getBytes());

        Assert.assertEquals(HttpStatus.SC_404_NOT_FOUND,
                HttpClientUtils.downloadStatus(httpClient, new HttpHead(urls.getPostUrl())));

        Assert.equals(HttpStatus.SC_202_ACCEPTED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutZippedAndRanged(urls.getPostUrl(), zippedPart1, 0, 7, text.length())));
        Assert.assertListsEqual(Cf.list("8"),
                HttpClientUtils.parseResponseHeaders(httpClient, new HttpHead(urls.getPostUrl()))
                        .getTs("Content-Length"));

        Assert.equals(HttpStatus.SC_201_CREATED, HttpClientUtils.downloadStatus(httpClient,
                UploaderHttpClientUtils.httpPutZippedAndRanged(urls.getPostUrl(), zippedPart2, 8, 9, 10)));

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());

        MulcaId mulcaFileId = afterUploadStatus.getFileMulcaId()
                .getOrThrow(afterUploadStatus.getXml().asXML());
        Assert.equals(text, mulcaClient.download(mulcaFileId).readText());

    }

    @Test
    public void failOnUploadFromBeginAfterResume() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        String text = "0123456789";

        // initial
        executeWithNetworkFail(httpClient,
                httpPutPartial(urls.getPostUrl(), text.substring(0, 5).getBytes(), 10));

        ThreadUtils.sleep(100);

        // upload from begin
        int code = urls.upload(InputStreamSourceUtils.bytes(text.getBytes()));
        Assert.assertEquals(HttpStatus.SC_400_BAD_REQUEST, code);
    }

    // iOS should success on resume upload from begin CHEMODAN-27279
    @Test
    public void successOnUploadFromBeginAfterResumeForiOS() throws Exception {
        String text = "0123456789";

        // initial
        UploaderClient.UploadUrls urls = startPartialUpload(text);

        // upload from begin
        String ua = "Yandex.Disk {\"os\":\"iOS\",\"src\":\"disk.mobile\",\"vsn\":\"1.70.2095\"," +
                "\"id\":\"some-id\",\"device\":\"phone\"}";
        int code = HttpClientUtils.downloadStatus(httpClient,
                HttpClientUtils.httpPut(urls.getPostUrl(), text.getBytes(), Cf.map("User-Agent", ua)));

        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);
    }

    @Test
    public void uploadJpegImageAndCheckPreview() throws Exception {
        doUploadImageAndCheckPreview("vini.jpg", Dimension.valueOf(720, 544), CHE_FILE);
    }

    @Test
    public void uploadIconImageAndCheckPreview() throws Exception {
        // for now, file extension is required for correct mime type detection for .ico
        doUploadImageAndCheckPreview("test.ico", Dimension.valueOf(16, 16),
                ChemodanFile.cons(UidOrSpecial.uid(PassportUid.cons(5181427L)), "234", "/test.ico"));
    }

    @Test
    public void uploadHeifImageAndCheckPreviewAndExif() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));

        int code = urls.upload(new ClassPathResourceInputStreamSource(ExifTool.class, "img.heic"),
                Cf.map("Content-Type", "image/heif"));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());
        Element generatePreviewEl = afterUploadStatus.getGenerateImageOnePreview();

        Assert.equals(ImageFormat.JPEG, ImageFormat.valueOf2(generatePreviewEl.attributeValue("format")));
        Assert.some(afterUploadStatus.getOnePreviewImageMulcaId());

        Assert.equals("2018-01-12T13:38:22Z", afterUploadStatus.getExifCreationDate());
        org.junit.Assert.assertEquals(59.958, Double.parseDouble(afterUploadStatus.getExifLatitude()), 0.001);
        org.junit.Assert.assertEquals(30.405, Double.parseDouble(afterUploadStatus.getExifLongitude()), 0.001);
    }

    private void doUploadImageAndCheckPreview(
            String previewImageRelPath, Dimension origSize, ChemodanFile chemodanFile)
    {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(chemodanFile);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        int code = urls.upload(new ClassPathResourceInputStreamSource(
                PreviewImageManagerTest.class, previewImageRelPath));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());
        Element generatePreviewEl = afterUploadStatus.getGenerateImageOnePreview();

        Assert.equals(ImageFormat.JPEG, ImageFormat.valueOf2(generatePreviewEl.attributeValue("format")));
        Assert.equals(origSize.getWidth(), Integer.parseInt(generatePreviewEl.attributeValue("width")));
        Assert.equals(origSize.getHeight(), Integer.parseInt(generatePreviewEl.attributeValue("height")));

        Assert.some(afterUploadStatus.getOnePreviewImageMulcaId());
    }

    @Test
    public void uploadImageAndCheckExifInfo() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        int code = urls.upload(new ClassPathResourceInputStreamSource(ExifTool.class, "small.jpg"));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());
        Assert.equals("2012-01-07T14:38:20Z", afterUploadStatus.getExifCreationDate());
        org.junit.Assert.assertEquals(59.959, Double.parseDouble(afterUploadStatus.getExifLatitude()), 0.001);
        org.junit.Assert.assertEquals(30.406, Double.parseDouble(afterUploadStatus.getExifLongitude()), 0.001);
    }

    @Test
    public void uploadVideoAndCheckVideoInfo() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        int code = urls.upload(new ClassPathResourceInputStreamSource(ChemodanFile.class, "mediainfo/small.mov"));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase(), afterUploadStatus.getStatus());

        Assert.equals("2013-03-07T14:52:21Z", afterUploadStatus.getMediaCreationDate());

        FileInformation fi = FileInformation.PS.getParser().parseJson(afterUploadStatus.getVideoInfo().get());
        Assert.equals("h264", fi.getVideoStream().getCodec());
        Assert.some(Dimension.valueOf(1280, 720), fi.getVideoStream().getDimension());
        Assert.equals(new DateTime(2013, 3, 7, 14, 52, 21, DateTimeZone.UTC).toInstant(), fi.getCreationTime().get());

        Assert.some(afterUploadStatus.getVideoPreviewMulcaId());
    }

    @Test
    public void uploadDocumentAndCheckPreview() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);
        UploaderClient.UploadStatusXml initialStatus = urls.requestStatusXml();
        Assert.assertFalse(initialStatus.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase()));
        Assert.none(initialStatus.getFileMulcaId());
        Assert.none(initialStatus.getDigestMulcaId());

        int code = urls.upload(new ClassPathResourceInputStreamSource(DocviewerClient.class, "test.pdf"));
        Assert.assertEquals(HttpStatus.SC_201_CREATED, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXmlUntilDone();
        Assert.some(afterUploadStatus.getDocumentPreviewMulcaId());
    }

    @Test
    public void headNotFoundRequest() {
        Assert.assertEquals(HttpStatus.SC_404_NOT_FOUND,
                HttpClientUtils.downloadStatus(httpClient,
                        new HttpHead("http://localhost:" + uploaderHttpPorts.getDataPort() + "/upload-target/fake-id")));
    }

    @Test
    public void uploadEmptyFileAndFailWithoutRetry() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        int code = urls.uploadMultipart(InputStreamSourceUtils.bytes(new byte[0]));
        Assert.assertEquals(HttpStatus.SC_503_SERVICE_UNAVAILABLE, code);

        UploaderClient.UploadStatusXml afterUploadStatus = urls.requestStatusXml();
        Assert.equals(UploadRequestStatus.Result.FAILED.name().toLowerCase(), afterUploadStatus.getStatus());
    }

    private UploaderClient.UploadUrls startPartialUpload(String text) throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        Assert.assertEquals(HttpStatus.SC_404_NOT_FOUND,
                HttpClientUtils.downloadStatus(httpClient, new HttpHead(urls.getPostUrl())));

        // initial
        executeWithNetworkFail(httpClient,
                httpPutPartial(urls.getPostUrl(), text.substring(0, 5).getBytes(), 10));

        ThreadUtils.sleep(100);
        return urls;
    }

    // CHEMODAN-34258: bug with content-type application/x-www-form-urlencoded
    @Test
    public void uploadFormUrlEncoded() throws Exception {
        UploaderClient.UploadUrls urls = uploaderClient.uploadToDisk(CHE_FILE);

        // org.eclipse.jetty.server.Request.maxFormContentSize default value 200000
        byte[] data = new byte[200100];

        MapF<String, String> headers = Cf.map("Content-Type", "application/x-www-form-urlencoded");
        Assert.equals(HttpStatus.SC_201_CREATED, HttpClientUtils.downloadStatus(httpClient,
                HttpClientUtils.httpPut(urls.getPostUrl(), data, headers)));

    }

}
