package ru.yandex.chemodan.uploader.web;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.registry.ZipFolderStages;
import ru.yandex.chemodan.uploader.web.client.UploaderClient;
import ru.yandex.chemodan.util.test.TestUser;
import ru.yandex.commune.archive.ArchiveEntry;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadBytesResponseHandler;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;
import static ru.yandex.chemodan.util.test.TestUser.ZIP_PUBLIC_FOLDER_HASH;

/**
 * @author ssytnik
 */
public class WebZipFolderTest extends AbstractWebTestSupport {
    @Autowired
    private ArchiveManager archiveManager;

    @Autowired
    @Qualifier("mulcaHttpClientForZipFolder")
    private HttpClient mulcaHttpClientForZipFolder;

    @Autowired
    private ZipFolderStages zipFolderStages;

    @Mock
    private HttpClient mulcaHttpClientForZipFolderMock;

    @Value("${mpfs.host}")
    private String mpfsHost;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void zipFolderPrivate() {
        checkZipFolder(getPrivateUri(TestUser.uid2.getUid(), "/disk/zipfoldertest"), "zipfoldertest", 306513);
    }

    @Test
    public void zipFolder404() throws IOException {
        String url = getPrivateUri(TestUser.uid2.getUid(), "/disk/zipfoldertest_test");
        HttpUriRequest request = new HttpGet(url);
        HttpClient client = ApacheHttpClientUtils.singleConnectionClient(Option.empty());
        HttpResponse response = client.execute(request);
        Assert.equals(404, response.getStatusLine().getStatusCode());
    }

    @Test
    public void zipFolderPublic() {
        checkZipFolder(getPublicUri(ZIP_PUBLIC_FOLDER_HASH), "zipfolderpublictest", 306657);
    }

    @Test
    public void zipFiles() throws UnsupportedEncodingException {
        String uri = UrlUtils.addParameter("http://" + mpfsHost + "/json/bulk_download_prepare",
                ApiArgs.UID, TestUser.uid2.getUid());
        StringEntity postingString =
                new StringEntity("{\"items\":[\"/disk/zipfoldertest/img1.png\",\"/disk/zipfoldertest/img2.png\"]}");

        HttpPost init = new HttpPost(uri);
        init.setEntity(postingString);
        init.setHeader("Content-type", "application/json");

        String response = ApacheHttpClientUtils.executeReadString(init, Timeout.seconds(30));

        JsonObject resp = (JsonObject) JsonParser.getInstance().parse(response);
        String oid = ((JsonString) resp.get("oid")).getString();

        checkZipFiles(getFilesUri(TestUser.uid2.getUid(), oid));
    }

    @Test
    public void shouldMakeRetriesWhenMulcaIsNotResponding() {
        withMock(() -> {
                try {
                    when(mulcaHttpClientForZipFolderMock.execute(any(HttpGet.class)))
                            .thenAnswer(withExceptionOnFirstCall());
                } catch (IOException e) {
                    throw ExceptionUtils.translate(e);
                }
                checkZipFolder(getPublicUri(ZIP_PUBLIC_FOLDER_HASH), "zipfolderpublictest", 306657);
            });
    }

    private Answer<HttpResponse> withExceptionOnFirstCall() {
        return new Answer<HttpResponse>() {

            private boolean wasCalled = false;

            @Override
            public HttpResponse answer(InvocationOnMock invocation) throws Throwable {
                if (!wasCalled) {
                    wasCalled = true;
                    throw new RuntimeException("Can't connect to mulca");
                } else {
                    return mulcaHttpClientForZipFolder.execute((HttpGet) invocation.getArguments()[0]);
                }
            }
        };
    }

    private void withMock(Function0V callback) {
        try {
            zipFolderStages.setMulcaHttpClientForZipFolder(mulcaHttpClientForZipFolderMock);
            callback.apply();
        } finally {
            zipFolderStages.setMulcaHttpClientForZipFolder(mulcaHttpClientForZipFolder);
        }
    }

    private String getPrivateUri(long uid, String path) {
        ChemodanFile chemodanFile = ChemodanFile.consWithoutFileId(
                UidOrSpecial.uid(PassportUid.cons(uid)), path);

        return UrlUtils.addParameters(UrlUtils.addParameter(
                "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.ZIP_FOLDER_URL,
                ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString()
                ), UploaderClient.toParameterMapWithoutFileId(chemodanFile));
    }

    private String getPublicUri(String hash) {
        return UrlUtils.addParameter(
                "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.ZIP_FOLDER_PUBLIC_URL,
                ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString(),
                ApiArgs.HASH, hash
                );
    }

    private String getFilesUri(long uid, String oid) {
        return UrlUtils.addParameter(
                "http://localhost:" + uploaderHttpPorts.getControlPort() + ApiUrls.ZIP_FILES_URL,
                ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString(),
                ApiArgs.UID, uid,
                ApiArgs.ZIP_FILES_MPFS_OID, oid
        );
    }

    private void checkZipFolder(String uri, String folderName, int fullLength) {
        byte[] out = downloadZip(uri);
        Assert.equals(fullLength, out.length);

        ArchiveListing listing = archiveManager.listArchive(InputStreamSourceUtils.bytes(out));
        Assert.equals(Cf.list(
                folderName, folderName + "/Empty", folderName + "/img1.png", folderName + "/img2.png",
                folderName + "/Вложенная", folderName + "/Вложенная/Warning.gif"),
                listing.getEntries().map(ArchiveEntry.getReadablePathF()).sorted());
        Assert.equals(0L, listing.getEntries().find(ArchiveEntry.pathEqualsToF(folderName + "/Вложенная")).get().getSize().get());
        Assert.equals(194929L, listing.getEntries().find(ArchiveEntry.pathEqualsToF(folderName + "/img1.png")).get().getSize().get());
    }

    private void checkZipFiles(String uri) {
        byte[] out = downloadZip(uri);
        Assert.equals(299786, out.length);

        ArchiveListing listing = archiveManager.listArchive(InputStreamSourceUtils.bytes(out));
        Assert.equals(Cf.list("archive", "archive/img1.png", "archive/img2.png"),
                listing.getEntries().map(ArchiveEntry.getReadablePathF()).sorted());
        Assert.equals(194929L, listing.getEntries().find(ArchiveEntry.pathEqualsToF("archive/img1.png")).get().getSize().get());
    }

    private byte[] downloadZip(String uri) {
        HttpUriRequest request = new HttpGet(uri);
        return ApacheHttpClientUtils.execute(request, new ReadBytesResponseHandler(), Timeout.seconds(20));
    }

}
