package ru.yandex.chemodan.app.urlshortener.api;


import java.io.OutputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.urlshortener.service.UrlShortenerManager;
import ru.yandex.chemodan.app.urlshortener.service.UrlShortenerUrlType;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.a3.action.result.SelfSerializer;
import ru.yandex.commune.a3.action.result.Variant;
import ru.yandex.commune.a3.action.result.Variants;
import ru.yandex.commune.a3.action.result.type.MediaType;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author conterouz
 */
@ActionContainer
public class UrlShortenerActions {

    private static final Pattern p = Pattern.compile("(([a-z]/)?[-_A-Za-z0-9/]+)[^A-Za-z0-9]?.*(\r.*)?");

    private final UrlShortenerManager manager;
    private final String webPrefix;
    private final String telemostPrefix;

    public UrlShortenerActions(UrlShortenerManager manager, String webPrefix, String telemostPrefix) {
        this.manager = manager;
        this.webPrefix = webPrefix;
        this.telemostPrefix = telemostPrefix;
    }

    @Action
    @WithMasterSlavePolicy(MasterSlavePolicy.R_ANY)
    // Anything that not stats with '--'
    @Path("{url:[^-][^-].*}")
    @Path("{url:[^-][^-].*\r.*}")
    public Object getUrl(
            @PathParam("url") String url,
            @SpecialParam ActionInvocationContext context)
    {
        String shortUrl = resolveShortUrl(url);

        Option<String> res = manager.resolve(shortUrl);
        String fullUrl = res.getOrThrow(() -> new NotFoundException("Could resolve short url: " + url));
        // manually set status and header Location:
        context.getHttpContext().setStatusCode(HttpStatus.SC_302_MOVED_TEMPORARILY);
        context.getHttpContext().setHeader("Location", fullUrl);

        return new SelfSerializer() {
            @Override
            protected void serialize(Variant variant, ActionInvocationContext context, OutputStream stream)
                    throws Exception
            {
                String result = new StringBuilder("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n" +
                        "<title>Redirecting...</title>\n" +
                        "<h1>Redirecting...</h1><p>You should be redirected automatically to target URL: <a href=\"")
                        .append(fullUrl).append("\">").append(fullUrl).append("</a>").toString();

               stream.write(result.getBytes());
               stream.flush();
            }

            @Override
            protected Variants getVariants() {
                return Variants.cons(MediaType.TEXT_HTML);
            }
        };
    }

    static String resolveShortUrl(String url) {
        Matcher matcher = p.matcher(url);
        if (!matcher.matches()) {
            throw new NotFoundException("Could resolve short url: " + url);
        }
        String shortUrl = matcher.group(1);
        shortUrl = StringUtils.removeEnd(shortUrl.trim(), "/").trim();
        return shortUrl;
    }

    @Action
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Path(value = "/--", methods = {HttpMethod.POST, HttpMethod.GET, HttpMethod.PUT})
    public Object generate(
            @RequestParam("url") String url,
            @RequestParam("type") String type,
            @SpecialParam ActionInvocationContext context)
    {
        return generateImpl(url, type);
    }

    @Action
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Path(value = "/", methods = {HttpMethod.POST, HttpMethod.PUT})
    public Object generateRoot(
            @RequestParam("url") String url,
            @RequestParam("type") String type,
            @SpecialParam ActionInvocationContext context)
    {
        return generateImpl(url, type);
    }

    private Object generateImpl(@RequestParam("url") String url, @RequestParam("type") String type) {
        UrlShortenerUrlType urlType = manager.parseType(type);
        String prefix;
        if (urlType == UrlShortenerUrlType.TELEMOST_JOIN) {
            prefix = telemostPrefix;
        } else {
            prefix = webPrefix;
        }

        String shortUrl = prefix + manager.generateShortUrl(url, urlType);

        return new SelfSerializer() {
            @Override
            protected void serialize(Variant variant, ActionInvocationContext context, OutputStream stream)
                    throws Exception
            {
                stream.write(shortUrl.getBytes());
                stream.flush();
            }

            @Override
            protected Variants getVariants() {
                return Variants.cons(MediaType.TEXT_HTML);
            }
        };
    }

}
