package ru.yandex.chemodan.app.urlshortener.api;

import java.io.UnsupportedEncodingException;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.chemodan.app.urlshortener.service.UrlShortenerManagerTest;
import ru.yandex.chemodan.app.urlshortener.test.ActivateUrlShortenerEmbeddedPg;
import ru.yandex.chemodan.app.urlshortener.test.TestBaseContextConfiguration;
import ru.yandex.chemodan.test.TestHelper;
import ru.yandex.chemodan.util.test.ApiTestBase;
import ru.yandex.commune.a3.action.result.ApplicationInfo;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author tolmalev
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
        TestBaseContextConfiguration.class,
        UrlShortenerApiContextConfiguration.class,
        UrlShortenerActionsTest.Context.class
})
@TestExecutionListeners(value = DependencyInjectionTestExecutionListener.class)
@ActivateUrlShortenerEmbeddedPg
public class UrlShortenerActionsTest extends ApiTestBase {
    @Value("${web.public.url}")
        String webPrefix;
    @Value("${telemost.public.url}")
        String telemostPrefix;

    static {
        TestHelper.initialize();
    }

    @Test
    public void goodNewUrls() throws UnsupportedEncodingException {
        UrlShortenerManagerTest.forAllPrefixAndTypes((prefix, type) -> {
            String url = prefix + Random2.R.nextAlnum(30);
            MockHttpServletResponse resp = sendRequest("GET", "/--?url=" + url + "&type=" + type.value());

            // wait for replication lag
            ThreadUtils.sleep(500);

            try {
                String shortHash = Uri2.parse(resp.getContentAsString()).getPath();
                validateGoodResponse(sendRequest("GET", shortHash), url);
            } catch (UnsupportedEncodingException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }

    @Test
    public void goodNewUrls2() throws UnsupportedEncodingException {
        UrlShortenerManagerTest.forAllPrefixAndTypes((prefix, type) -> {
            String url = prefix + Random2.R.nextAlnum(30);
            MockHttpServletResponse resp = sendRequest("POST", "?url=" + url + "&type=" + type.value());

            // wait for replication lag
            ThreadUtils.sleep(500);

            try {
                String shortHash = Uri2.parse(resp.getContentAsString()).getPath();
                validateGoodResponse(sendRequest("GET", shortHash), url);
                validateGoodResponse(sendRequest("GET", shortHash + "/"), url);
                validateGoodResponse(sendRequest("GET", shortHash + " \r/"), url);
                validateGoodResponse(sendRequest("GET", shortHash + " \r2/"), url);
            } catch (UnsupportedEncodingException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }

    @Test
    public void badType() {
        String url = Random2.R.nextAlnum(10);
        MockHttpServletResponse resp = sendRequest("GET", "/--?url=" + url + "&type=sdfhgasdfjgadskfjh");
        Assert.equals(500, resp.getStatus());
    }

    @Test
    public void noType() {
        String url = Random2.R.nextAlnum(10);
        MockHttpServletResponse resp = sendRequest("GET", "/--?url=" + url);
        Assert.equals(400, resp.getStatus());
    }

    @Test
    public void badUrl() {
        MockHttpServletResponse resp = sendRequest("GET", "/blagfjgjghggsla");
        Assert.equals(404, resp.getStatus());
    }

    @Test
    public void webDomain() {
        MockHttpServletResponse resp = sendRequest("POST", "?url=ST5bUOkp37vtgk&type=yadisk");
        // wait for replication lag
        ThreadUtils.sleep(500);
        try {
            Assert.isTrue(resp.getContentAsString().startsWith(webPrefix));
        } catch (UnsupportedEncodingException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Test
    public void telemostDomain() {
        MockHttpServletResponse resp = sendRequest("POST", "?url=ST5bUOkp37vtgk&type=telemost-join");
        // wait for replication lag
        ThreadUtils.sleep(500);
        try {
            Assert.isTrue(resp.getContentAsString().startsWith(telemostPrefix));
        } catch (UnsupportedEncodingException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Test
    public void resolveShortUrl() {
        Assert.equals("d/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk/"));
        Assert.equals("d/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk)"));
        Assert.equals("d/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk="));
        Assert.equals("d/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk?force_show=1"));
        Assert.equals("d/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk/?force_show=1"));

        Assert.equals("d/S--T__5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("d/S--T__5bUOkp37vtgk/"));

        Assert.equals("d/ST5bUOkp37vtgk/ads", UrlShortenerActions.resolveShortUrl("d/ST5bUOkp37vtgk/ads"));
        Assert.equals("j/ST5bUOkp37vtgk", UrlShortenerActions.resolveShortUrl("j/ST5bUOkp37vtgk"));
    }

    private void validateGoodResponse(MockHttpServletResponse resp, String rightUrl) {
        Assert.equals(302, resp.getStatus());
        Assert.equals(rightUrl, resp.getHeader("Location"));
    }

    @Configuration
    static class Context {
        @Bean
        public ApplicationInfo applicationInfo(AppName appName) {
            return new ApplicationInfo(appName.appName(), "tests");
        }
    }
}
