package ru.yandex.chemodan.app.urlshortener.service;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.urlshortener.dao.UrlShortenerJdbcDao;
import ru.yandex.chemodan.app.urlshortener.dao.UrlShortenerMappingItem;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author conterouz
 */
public class UrlShortenerManager implements PingerChecker {
    private MapF<Short, String> prefixes = null;

    private final UrlShortenerJdbcDao dao;
    private final Duration linkTtl;

    public UrlShortenerManager(UrlShortenerJdbcDao dao, Duration linkTtl) {
        this.dao = dao;
        this.linkTtl = linkTtl;
    }

    public Option<Instant> generateEolTimeByType(UrlShortenerUrlType urlType) {
        switch (urlType) {
            case TELEMOST_JOIN:
                return Option.of(Instant.now().plus(linkTtl));
            default:
                return Option.empty();
        }
    }

    public UrlShortenerUrlType parseType(String type) {
        return UrlShortenerUrlType.R.fromValueO(type).getOrThrow(
                () -> new IllegalArgumentException("Type " + type + " is unsupported")
        );
    }

    public String generateShortUrl(String url, UrlShortenerUrlType urlType) {
        Tuple2<Short, String> prefix = prefixes.entries()
                .findBy2(url::startsWith)
                .getOrElse(Tuple2.tuple((short) 0, ""));

        url = url.substring(prefix._2.length());

        while (true) {
            String shortUrl = urlType.getShortUrl();
            UrlShortenerMappingItem urlShortenerMappingItem = UrlShortenerMappingItem.builder()
                    .shortUrl(shortUrl)
                    .prefixId(prefix._1)
                    .url(url)
                    .eolTime(generateEolTimeByType(urlType))
                    .build();

            if (dao.insert(urlShortenerMappingItem)) {
                return shortUrl;
            }

            if (urlType == UrlShortenerUrlType.TELEMOST_JOIN) {
                boolean isLinkExpired = dao
                        .find(shortUrl)
                        .filterMap(UrlShortenerMappingItem::getEolTime)
                        .exists(Instant::isBeforeNow);
                if (isLinkExpired && dao.update(urlShortenerMappingItem)) {
                    return shortUrl;
                }
            }
        }
    }

    public Option<String> resolve(String shortUrl) {
        // resolve both link starting with slash and not
        String finalShortUrl = StringUtils.removeStart(shortUrl, "/");
        return dao
                .find(finalShortUrl)
                .filterNot(info -> info.getEolTime().exists(Instant::isBeforeNow))
                .map(t -> prefixes.getOrElse(t.getPrefixId(), "") + t.getUrl());
    }

    public void setPrefixes(MapF<Short, String> prefixes) {
        // zero prefix id is denied
        this.prefixes = prefixes.filterKeys(id -> id != 0);
    }

    @Override
    public boolean isActive() {
        return prefixes != null;
    }
}
