package ru.yandex.chemodan.app.urlshortener.service;

import java.util.function.BiConsumer;
import java.util.regex.Pattern;

import org.joda.time.Instant;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.support.DependencyInjectionTestExecutionListener;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.urlshortener.dao.UrlShortenerJdbcDao;
import ru.yandex.chemodan.app.urlshortener.dao.UrlShortenerMappingItem;
import ru.yandex.chemodan.app.urlshortener.test.ActivateUrlShortenerEmbeddedPg;
import ru.yandex.chemodan.app.urlshortener.test.TestBaseContextConfiguration;
import ru.yandex.chemodan.test.TestHelper;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;


/**
 * @author tolmalev
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
        TestBaseContextConfiguration.class
})
@TestExecutionListeners(value = DependencyInjectionTestExecutionListener.class)
@ActivateUrlShortenerEmbeddedPg
public class UrlShortenerManagerTest {
    static {
        TestHelper.initialize();
    }

    public static final ListF<String> PREFIXES = Cf
            .list("",
                    "https://disk.yandex.net/disk/public/?hash=",
                    "https://disk-test.disk.yandex.ru/public/?hash=",
                    "https://disk.yandex.net/albums/public/?key="
            );

    public static void forAllPrefixAndTypes(BiConsumer<String, UrlShortenerUrlType> consumer) {
        for (UrlShortenerUrlType type : UrlShortenerUrlType.values()) {
            for (String prefix : PREFIXES) {
                consumer.accept(prefix, type);
            }
        }
    }

    @Autowired
    private UrlShortenerManager manager;
    @Autowired
    private UrlShortenerJdbcDao dao;

    @Test
    public void newLinks() {
        forAllPrefixAndTypes((prefix, type) -> {
            String url = prefix + Random2.R.nextAlnum(30);
            String shortUrl = manager.generateShortUrl(url, type);

            Assert.isFalse(url.startsWith("/"), "link mustn't start with /");
            testResolve(url, shortUrl);
        });
    }

    @Test
    public void badLink() {
        Assert.none(manager.resolve("asfdjhadsfjkgqo4tu2"));
    }

    @Test(expected = IllegalArgumentException.class)
    public void badType() {
        manager.parseType("unknown_type");
    }

    @Test(expected = IllegalArgumentException.class)
    public void emptyType() {
        manager.parseType("");
    }

    @Test
    public void emptyUrl() {
        String shortUrl = manager.generateShortUrl("", UrlShortenerUrlType.YA_DISK);
        Assert.some("", manager.resolve(shortUrl));
    }

    @Test
    public void isActiveAfterContextInit() {
        Assert.isTrue(manager.isActive());
    }

    @Test
    public void telemostUrl() {
        String shortUrl = manager.generateShortUrl("test", UrlShortenerUrlType.TELEMOST_JOIN);
        testResolve("test", shortUrl);
        Assert.isTrue(
                Pattern.matches("^" + UrlShortenerUrlType.TELEMOST_JOIN.getLetter() + "/[0-9]{14}$", shortUrl),
                shortUrl
        );
    }

    private String generateSameUrl(String url, UrlShortenerUrlType urlType) {
        Random2.R.random.setSeed(123);
        return manager.generateShortUrl(url, urlType);
    }

    @Test
    public void collideTelemostUrl() {
        String shortUrl = generateSameUrl("test", UrlShortenerUrlType.TELEMOST_JOIN);
        testResolve("test", shortUrl);

        String shortUrl2 = generateSameUrl("test2", UrlShortenerUrlType.TELEMOST_JOIN);
        Assert.notEquals(shortUrl, shortUrl2, "First link must be actual");

        dao.update(UrlShortenerMappingItem.builder()
                .shortUrl(shortUrl)
                .prefixId((short) 0)
                .url("test")
                .eolTime(Option.of(Instant.now().minus(2 * 365L * 24 * 3600 * 1000)))
                .build()
        );
        String shortUrl3 = generateSameUrl("test3", UrlShortenerUrlType.TELEMOST_JOIN);
        Assert.equals(shortUrl, shortUrl3, "Link is obsolete now, must rewrite");
        testResolve("test3", shortUrl);
    }

    @Test
    public void collideDiskUrl() {
        String shortUrl = generateSameUrl("test", UrlShortenerUrlType.YA_DISK);
        testResolve("test", shortUrl);

        String shortUrl2 = generateSameUrl("test2", UrlShortenerUrlType.YA_DISK);
        Assert.notEquals(shortUrl, shortUrl2, "Ya Disk does not collide");

        Assert.isEmpty(dao.find(shortUrl).map(UrlShortenerMappingItem::getEolTime).get());
        testResolve("test", shortUrl);
    }

    private void testResolve(String url, String shortUrl) {
        Assert.equals(Option.of(url), manager.resolve(shortUrl), shortUrl + " must be resolved into " + url);
        Assert.equals(Option.of(url), manager.resolve("/" + shortUrl),
                "/" + shortUrl + " must be resolved into " + url);
    }
}
