package ru.yandex.chemodan.app.videostreaming;

import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.encrypt.OpenSslAes256CbcCrypter;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsManager;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsUrlBuilder;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingUrlParams;
import ru.yandex.chemodan.videostreaming.framework.web.HlsServlet;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DiskStreamingUrlBuilder implements HlsUrlBuilder<MpfsSourceMeta> {
    private static final SetF<String> VALID_QS_PARAMS = Cf.set(
            "disable_cache", "delete_cache", "init_prefetch_size", "ffmpeg_out_threads", "segment_duration");

    private static final SetF<String> SKIP_ENCODING_QS_PARAMS = Cf.set("stid");

    private final Duration linkLifetime;

    private final OpenSslAes256CbcCrypter crypter;

    private final MapF<Option<URI>, URI> balancerUriMapping;

    private final Option<URI> defaultBalancerUri;

    public DiskStreamingUrlBuilder(OpenSslAes256CbcCrypter crypter, MapF<Option<URI>, URI> balancerUriMapping,
            Duration linkLifetime)
    {
        this.crypter = crypter;
        this.balancerUriMapping = balancerUriMapping;
        this.defaultBalancerUri = balancerUriMapping.values().iterator().nextO();
        this.linkLifetime = linkLifetime;
    }

    public URI getSegmentPrefetchUrl(HlsSegmentMeta<MpfsSourceMeta> segmentMeta) {
        return getSegmentPrefetchUrl(segmentMeta, StreamingUrlParams.getCurrent());
    }

    public URI getSegmentPrefetchUrl(HlsSegmentMeta<MpfsSourceMeta> segmentMeta, StreamingUrlParams urlParams) {
        return getSegmentUrl(segmentMeta, urlParams, "prefetch");
    }

    public URI getLocalSegmentPrefetchUrl(HlsSegmentMeta<MpfsSourceMeta> segmentMeta) {
        return getLocalSegmentPrefetchUrl(segmentMeta, StreamingUrlParams.getCurrent());
    }

    public URI getLocalSegmentPrefetchUrl(HlsSegmentMeta<MpfsSourceMeta> segmentMeta, StreamingUrlParams urlParams) {
        return getSegmentUrl(segmentMeta, urlParams, "local");
    }

    private URI getSegmentUrl(HlsSegmentMeta<MpfsSourceMeta> segmentMeta, StreamingUrlParams urlParams, String prefix) {
        return new Builder(urlParams)
                .setPath(urlParams.getPathO().get())
                .replaceFilename(prefix + segmentMeta.getIndex() + ".ts")
                .build();
    }

    @Override
    public URI getMasterPlaylistUrl(MpfsSourceMeta sourceMeta, StreamingUrlParams urlParams) {
        return new Builder(urlParams)
                .setPath(HlsServlet.URL_PREFIX)
                .addPathSecrets(sourceMeta)
                .addPath(HlsManager.MASTER_PLAYLIST_FILENAME)
                .build();
    }

    @Override
    public URI getPlaylistUrl(MpfsSourceMeta sourceMeta, HlsStreamQuality quality, StreamingUrlParams urlParams) {
        return new Builder(urlParams)
                .setPath(HlsServlet.URL_PREFIX)
                .addPathSecrets(sourceMeta)
                .addPath(quality.toRequestParamValue())
                .addPath(HlsManager.PLAYLIST_FILENAME)
                .build();
    }

    @Override
    public String getRelativePlaylistItemUri(String filename, Tuple2List<String, String> extraParams) {
        String qsSuffix = UrlUtils.listMapToUrlParameters(
                StreamingUrlParams.getCurrent().getParameters()
                        .plus(extraParams)
        );
        return filename + (StringUtils.isNotBlank(qsSuffix) ? "?" : "") + qsSuffix;
    }

    class Builder {
        final URI balancerUri;

        Path path;

        Tuple2List<String, String> parameters;

        final boolean useHttp;

        Builder(StreamingUrlParams urlParams) {
            this.balancerUri = balancerUriMapping.getO(urlParams.getServerUriO())
                    .orElse(urlParams.getServerUriO())
                    .orElse(defaultBalancerUri)
                    .getOrThrow("Could not determine host");
            this.path = Paths.get("/");
            this.parameters = urlParams.getParameters().filterBy1(VALID_QS_PARAMS::containsTs);
            this.useHttp = urlParams.isUseHttp();
        }

        Builder setPath(String pathStr) {
            path = Paths.get(pathStr);
            return this;
        }

        Builder addPath(String other) {
            path = path.resolve(other);
            return this;
        }

        Builder replaceFilename(String other) {
            path = path.resolveSibling(other);
            return this;
        }

        @SuppressWarnings("unused")
        Builder addParamIfMissing(@SuppressWarnings("SameParameterValue") String name, String value) {
            if (!parameters.get1().containsTs(name)) {
                parameters = parameters.plus1(name, value);
            }
            return this;
        }

        Builder addPathSecrets(MpfsSourceMeta sourceMeta) {
            String encryptedSourceMeta = crypter.encrypt(sourceMeta.serialize());
            String timestamp = Long.toHexString(getExpireTimeMicros());
            String sign = crypter.sign(encryptedSourceMeta + "/" + timestamp);
            return addPath(encryptedSourceMeta)
                    .addPath(timestamp)
                    .addPath(sign);
        }

        URI build() {
            return UrlUtils.uri(getBaseUrl() + getQueryStringSuffix());
        }

        String getBaseUrl() {
            UriBuilder uriBuilder = new UriBuilder(balancerUri)
                    .appendPath(getPath());
            if (useHttp) {
                uriBuilder.setScheme("http");
            }
            return uriBuilder.toUrl();
        }

        String getPath() {
            return path.toString();
        }

        String getQueryStringSuffix() {
            return (parameters.isNotEmpty() ? "?" : "")
                    + parameters.map(DiskStreamingUrlBuilder::toQueryParameter)
                    .mkString("&");
        }
    }

    private long getExpireTimeMicros() {
        return (System.currentTimeMillis() + linkLifetime.getMillis()) * 1000;
    }

    private static String toQueryParameter(String param, String value) {
        return !SKIP_ENCODING_QS_PARAMS.containsTs(param)
                ? UrlUtils.toQueryParameter(param, value)
                : param + "=" + value;
    }
}
