package ru.yandex.chemodan.app.videostreaming;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Value;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithIp;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithKey;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithTranscodeParams;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.InstantAsMillisMarshaller;
import ru.yandex.misc.bender.custom.InstantAsMillisUnmarshaller;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.ip.bender.IpAddressMarshaller;
import ru.yandex.misc.ip.bender.IpAddressUnmarshaller;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author vavinov
 * @author Dmitriy Amelin (lemeh)
 */
@Value
@AllArgsConstructor
@Builder(toBuilder = true)
@BenderBindAllFields
public class MpfsSourceMeta implements SourceMetaWithIp, SourceMetaWithKey, SourceMetaWithTranscodeParams {
    private static final BenderParserSerializer<MpfsSourceMeta> parserSerializer =
            BenderParserSerializer.cons(
                    ClassX.wrap(MpfsSourceMeta.class),
                    new BenderConfiguration(
                            BenderConfiguration.defaultSettings(),
                            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                                    .add(IpAddress.class, new IpAddressMarshaller(), new IpAddressUnmarshaller())
                                    .add(Instant.class, new InstantAsMillisMarshaller(),
                                            new InstantAsMillisUnmarshaller())
                                    .build()
                    )
            );

    MulcaId mulcaId;

    Option<PassportUidOrZero> ownerUid;

    Option<PassportUidOrZero> consumerUid;

    boolean publicUser;

    Option<IpAddress> userIp;

    Option<Instant> expireAt;

    Option<String> argFiller;

    Option<String> ffVer;

    public MpfsSourceMeta(MulcaId mulcaId) {
        this(mulcaId, Option.empty(), Option.empty(), true, Option.empty(), Option.empty(), Option.empty(),
                Option.empty());
    }

    public MpfsSourceMeta(MulcaId mulcaId, Option<PassportUidOrZero> ownerUid, Option<PassportUidOrZero> consumerUid,
            boolean publicUser, Option<IpAddress> userIp)
    {
        this(mulcaId, ownerUid, consumerUid, publicUser, userIp, Option.empty(), Option.empty(), Option.empty());
    }

    public static MpfsSourceMeta parse(String str) {
        try {
            return parserSerializer.getParser().parseJson(str);
        } catch (RuntimeIoException ex) {
            if (ex.getCause() instanceof com.fasterxml.jackson.core.JsonParseException) {
                return new MpfsSourceMeta(MulcaId.fromSerializedString(str));
            }
            throw ex;
        }
    }

    public String getStidStr() {
        return mulcaId.toSerializedString();
    }

    @Override
    public String getKey() {
        return getStidStr();
    }

    public String serialize() {
        return new String(
                parserSerializer.getSerializer()
                        .serializeJson(this),
                CharsetUtils.UTF8_CHARSET
        );
    }
}
