package ru.yandex.chemodan.app.videostreaming;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.encrypt.OpenSslAes256CbcCrypter;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.util.RequestAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.web.HlsServlet;
import ru.yandex.chemodan.videostreaming.framework.web.StreamingHttpServletRequest;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MpfsSourceMetaParser implements SourceMetaParser<MpfsSourceMeta> {
    private static final Logger logger = LoggerFactory.getLogger(MpfsSourceMetaParser.class);

    private final OpenSslAes256CbcCrypter crypter;

    private final RequestAccessChecker accessChecker;

    public MpfsSourceMetaParser(OpenSslAes256CbcCrypter crypter, RequestAccessChecker accessChecker) {
        this.crypter = crypter;
        this.accessChecker = accessChecker;
    }

    @Override
    public MpfsSourceMeta parse(StreamingHttpServletRequest request) {
        return parse(request.getPathInfo(), accessChecker.isFromExternalHost(request));
    }

    public MpfsSourceMeta parse(String uri, boolean checkExpireDate) {
        ListF<String> segments = Cf.x(Paths.get(uri).iterator())
                .map(Path::toString)
                .map(UrlUtils::urlDecode)
                .toList();

        if (HlsServlet.URL_PREFIX.equals(segments.first())) {
            segments = segments.drop(1);
        }

        return parse(checkExpireDate, segments.get(0), segments.get(1), segments.get(2));
    }

    private MpfsSourceMeta parse(boolean checkExpireDate, String encryptedData, String timestampHex, String sign) {
        MpfsSourceMeta srcMeta = MpfsSourceMeta.parse(crypter.decrypt(encryptedData));

        if (!sign.equals(crypter.sign(encryptedData + "/" + timestampHex))) {
            logger.error("Incorrect sign specified - probable attack");
            throw new AccessForbiddenException(srcMeta);
        }

        Instant expireDate = new Instant(Long.parseUnsignedLong(timestampHex, 16) / 1000);
        if (checkExpireDate && Instant.now().isAfter(expireDate)) {
            logger.warn("Requested expired link");
            throw new LinkExpiredException(srcMeta);
        }

        return srcMeta.toBuilder()
                .expireAt(Option.of(expireDate))
                .build();
    }
}
