package ru.yandex.chemodan.app.videostreaming;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.joda.time.Days;
import org.joda.time.Instant;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.encrypt.OpenSslAes256CbcCrypter;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.util.RequestAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.web.StreamingHttpServletRequest;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MpfsSourceMetaParserTest {
    private static final String VALID_PASSWORD = "password";

    private static final String INVALID_PASSWORD = "wrong-password";

    private static final String STID = "320.yadisk:4003323001.E1614:297259517224825672142543749822";

    private static final RequestAccessChecker DEFAULT_ACCESS_CHECKER = new RequestAccessChecker(Cf.set(), Cf.set());

    @Test
    public void testParse() {
        testParse(Instant.now().plus(Days.days(1).toStandardDuration()), VALID_PASSWORD);
    }

    @Test(expected = SourceMetaParser.AccessForbiddenException.class)
    public void testParseWithInvalidHmac() {
        testParse(Instant.now().plus(Days.days(1).toStandardDuration()), INVALID_PASSWORD);
    }

    @Test(expected = SourceMetaParser.LinkExpiredException.class)
    public void testParseWithExpiredDate() {
        testParseWithExpiredDate(DEFAULT_ACCESS_CHECKER);
    }

    @Test
    public void testParseWithExpiredDateWithInternalHost() {
        testParseWithExpiredDate(new RequestAccessChecker(Cf.set("host.org"), Cf.set()));
    }

    private void testParseWithExpiredDate(RequestAccessChecker accessChecker) {
        testParse(Instant.now().minus(Days.days(1).toStandardDuration()), VALID_PASSWORD, accessChecker);
    }

    private static void testParse(Instant expireDate, String origPassword) {
        testParse(expireDate, origPassword, DEFAULT_ACCESS_CHECKER);
    }

    private static void testParse(Instant expireDate, String origPassword, RequestAccessChecker accessChecker) {
        String timestampHex = Long.toHexString(expireDate.getMillis() * 1000);

        Path path = Paths.get(STID, timestampHex, origPassword.equals(VALID_PASSWORD) ? "sign" : "invalidsign");
        StreamingHttpServletRequest request = Mockito.mock(StreamingHttpServletRequest.class);
        Mockito.when(request.getServerName())
                .thenReturn("host.org");
        Mockito.when(request.getPathInfo())
                .thenReturn(path.toString());
        Mockito.when(request.getXRealIp())
                .thenReturn(Option.empty());

        OpenSslAes256CbcCrypter crypter = Mockito.mock(OpenSslAes256CbcCrypter.class);
        Mockito.when(crypter.encrypt(Mockito.anyString()))
                .thenAnswer(invocation -> origPassword + invocation.getArgument(0));
        Mockito.when(crypter.decrypt(Mockito.anyString()))
                .thenAnswer(inv ->
                        inv.<String>getArgument(0)
                                .replaceAll("^" + VALID_PASSWORD, "")
                );
        Mockito.when(crypter.sign(Mockito.anyString()))
                .thenReturn("sign");

        Assert.equals(
                new MpfsSourceMeta(MulcaId.fromSerializedString(STID)),
                new MpfsSourceMetaParser(crypter, accessChecker)
                        .parse(request).toBuilder()
                        .expireAt(Option.empty())
                        .build()
        );
    }

}
