package ru.yandex.chemodan.app.videostreaming;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.overrides.TranscodeOverride;
import ru.yandex.chemodan.app.videostreaming.overrides.TranscodeOverridesRegistry;
import ru.yandex.chemodan.videostreaming.framework.config.FFmpegVersions;
import ru.yandex.chemodan.videostreaming.framework.hls.ForceFFmpeg3ForFormats;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsResource;
import ru.yandex.chemodan.videostreaming.framework.hls.TranscodeParams;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaTransformer;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.AbstractPassportUid;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MpfsSourceMetaTransformer implements SourceMetaTransformer<MpfsSourceMeta> {
    private static final DynamicProperty<Integer> nextFFmpegParamsPercentage =
            DynamicProperty.cons("streaming-next-ffmpeg-params-percentage", 0);

    private final TranscodeOverridesRegistry overridesRegistry;

    private final ForceFFmpeg3ForFormats forceFFmpeg3ForFormats;

    private final TransitionalValue<String> argFiller;

    private final TransitionalValue<String> ffmpegVersion;

    public MpfsSourceMetaTransformer(
            TranscodeOverridesRegistry overridesRegistry,
            ForceFFmpeg3ForFormats forceFFmpeg3ForFormats,
            TransitionalValue<String> argFiller,
            TransitionalValue<String> ffmpegVersion)
    {
        this.overridesRegistry = overridesRegistry;
        this.forceFFmpeg3ForFormats = forceFFmpeg3ForFormats;
        this.argFiller = argFiller;
        this.ffmpegVersion = ffmpegVersion;
    }

    @Override
    public MpfsSourceMeta transform(MpfsSourceMeta sourceMeta, HlsResource resource) {
        Option<TranscodeParams> overriddenParams = overridesRegistry.getOverrideO(sourceMeta)
                .map(TranscodeOverride::getParams);
        String argFillerName = overriddenParams.map(TranscodeParams::getArgFillerName)
                .getOrElse(() -> getDefaultArgFiller(sourceMeta));
        String ffmpegVersion = overriddenParams.filterMap(TranscodeParams::getFFmpegVersionO)
                .getOrElse(() -> getFFmpegVersion(sourceMeta, resource));
        return sourceMeta.toBuilder()
                .argFiller(Option.of(argFillerName))
                .ffVer(Option.of(ffmpegVersion))
                .build();
    }

    private String getDefaultArgFiller(MpfsSourceMeta sourceMeta) {
        return isNextParamsEnabled(sourceMeta) ? argFiller.getNext() : argFiller.getCurrent();
    }

    private String getFFmpegVersion(MpfsSourceMeta sourceMeta, HlsResource resource) {
        String format = resource.fileInformation.getFormat();
        if (forceFFmpeg3ForFormats.contains(format)) {
            return FFmpegVersions.VER_3_2;
        }
        return isNextParamsEnabled(sourceMeta) ? ffmpegVersion.getNext() : ffmpegVersion.getCurrent();
    }

    private static boolean isNextParamsEnabled(MpfsSourceMeta sourceMeta) {
        return sourceMeta.getOwnerUid()
                .map(AbstractPassportUid::getUid)
                .isMatch(MpfsSourceMetaTransformer::isNextParamsEnabledForUid);
    }

    private static boolean isNextParamsEnabledForUid(Long uid) {
        return Math.abs(uid % 100) < nextFFmpegParamsPercentage.get();
    }
}
