package ru.yandex.chemodan.app.videostreaming;

import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.web.StreamingHttpServletRequest;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.concurrent.TimeoutRuntimeException;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.ip.Ipv4Address;
import ru.yandex.misc.ip.Ipv6Address;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.regex.Pattern2;

public class VideoInfoMpfsSourceMetaParser implements SourceMetaParser<MpfsSourceMeta> {
    private static final Logger logger = LoggerFactory.getLogger(VideoInfoMpfsSourceMetaParser.class);

    private static final SetF<IpAddress> EXCLUDE_IP_ADDRESSES = Cf.set(
            Ipv4Address.ANY_ADDRESS, Ipv6Address.ANY_ADDRESS, Ipv4Address.LOOPBACK, Ipv6Address.LOOPBACK
    );

    private static final Pattern2 YANDEX_HOSTNAME_PATTERN =
            Pattern2.compile(".+\\.yandex\\.(net|ru)", Pattern.CASE_INSENSITIVE);

    private static final DynamicProperty<ListF<String>> trustedClientIds =
            DynamicProperty.cons("streaming-trusted-client-ids", Cf.list());

    @Override
    public MpfsSourceMeta parse(StreamingHttpServletRequest req) {
        MulcaId stid = req.getParameterO("stid")
                .orElse(() -> req.getParameterO("source"))
                .map(MulcaId::fromSerializedString)
                .getOrThrow(SourceNotSpecifiedException::new);
        Option<PassportUidOrZero> ownerUid = req.getParameterO("owner_uid")
                .map(Long::parseLong)
                .map(PassportUidOrZero::fromUid);
        Option<PassportUidOrZero> consumerUid = req.getParameterO("consumer_uid")
                .map(Long::parseLong)
                .map(PassportUidOrZero::fromUid);
        boolean isPublic = req.getParameterO("public")
                .map(Boolean::parseBoolean)
                .getOrElse(true);
        Option<String> clientId = req.getParameterO("client_id");
        Option<IpAddress> userIp = req.getParameterO("user_ip")
                .map(IpAddress::parse)
                .filter(this::isValidUserIp)
                .filterNot(ipAddress -> clientId.isMatch(VideoInfoMpfsSourceMetaParser::isTrustedClient));
        return new MpfsSourceMeta(stid, ownerUid, consumerUid, isPublic, userIp);
    }

    private static boolean isTrustedClient(String clientId) {
        return trustedClientIds.get().containsTs(clientId);
    }

    private boolean isValidUserIp(IpAddress ipAddress) {
        try {
            return !EXCLUDE_IP_ADDRESSES.containsTs(ipAddress);
        } catch (TimeoutRuntimeException ex) {
            logger.warn("Timeout while waiting reverse DNS lookup");
            return false;
        }
    }

    static boolean isYandexHostname(String hostname) {
        return YANDEX_HOSTNAME_PATTERN.matches(hostname);
    }
}
