package ru.yandex.chemodan.app.videostreaming;

import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.web.StreamingHttpServletRequest;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.ip.Ipv4Address;
import ru.yandex.misc.ip.Ipv6Address;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class VideoInfoMpfsSourceMetaParserTest {
    private static final SetF<String> VALID_PARAMS =
            Cf.set("stid", "source", "owner_uid", "consumer_uid", "public", "user_ip");

    private static final String VALID_STID = "320.yadisk:4003323001.E1614:297259517224825672142543749822";

    private static final IpAddress YANDEX_IP = IpAddress.parse("213.180.204.12");

    private static final IpAddress NON_YANDEX_IP = IpAddress.parse("2.2.2.2");

    private static final IpAddress NON_YANDEX_IP_WITH_FQDN = IpAddress.parse("3.3.3.3");

    @Test(expected = SourceMetaParser.SourceNotSpecifiedException.class)
    public void parserThrowsExceptionWhenSourceNotSpecified() {
        consSut().parse(consRequest());
    }

    @Test
    public void parserMustAcceptZeroUids() {
        Assert.equals(
                new MpfsSourceMeta(
                        MulcaId.fromSerializedString(VALID_STID),
                        Option.of(PassportUidOrZero.zero()),
                        Option.of(PassportUidOrZero.zero()),
                        true,
                        Option.empty()
                ),
                consSut().parse(consRequest("stid", VALID_STID, "consumer_uid", "0", "owner_uid", "0"))
        );
    }

    @Test
    public void parserMustParseCorrectlyWhenAllParamsSpecified() {
        testParse(NON_YANDEX_IP, Option.of(NON_YANDEX_IP));
    }

    @Test
    public void parserMustNotFilterOutYandexIps() {
        testParse(YANDEX_IP, Option.of(YANDEX_IP));
    }

    @Test
    public void parserMustNotFilterOutNonYandexIpWithFqdn() {
        testParse(NON_YANDEX_IP_WITH_FQDN, Option.of(NON_YANDEX_IP_WITH_FQDN));
    }

    @Test
    public void parserMustFilterOutIpv4Loopback() {
        testParse(Ipv4Address.LOOPBACK, Option.empty());
    }

    @Test
    public void parserMustFilterOutIpv6Loopback() {
        testParse(Ipv6Address.LOOPBACK, Option.empty());
        testParse(Ipv6Address.LOOPBACK, Option.empty());
    }

    @Test
    public void parserMustFilterOutIpv4Any() {
        testParse(Ipv4Address.ANY_ADDRESS, Option.empty());
        testParse(Ipv6Address.LOOPBACK, Option.empty());
    }

    @Test
    public void parserMustFilterOutIpv6Any() {
        testParse(Ipv6Address.ANY_ADDRESS, Option.empty());
    }

    private void testParse(IpAddress inputIpAddress, Option<IpAddress> expectedIpAddress) {
        Assert.equals(
                new MpfsSourceMeta(
                        MulcaId.fromSerializedString(VALID_STID),
                        Option.of(PassportUidOrZero.fromUid(123)),
                        Option.of(PassportUidOrZero.fromUid(1234567890)),
                        false,
                        expectedIpAddress
                ),
                consSut()
                        .parse(consRequest(
                                "stid", VALID_STID,
                                "consumer_uid", "1234567890",
                                "owner_uid", "123",
                                "public", "False",
                                "user_ip", inputIpAddress.format())
                        )
        );
    }


    private static StreamingHttpServletRequest consRequest(Object... paramValues) {
        return consRequest(Tuple2List.fromPairs(paramValues));
    }

    private static StreamingHttpServletRequest consRequest(Tuple2List<String, String> paramValues) {
        StreamingHttpServletRequest request = Mockito.mock(StreamingHttpServletRequest.class);
        Mockito.when(request.getParameterO(Mockito.anyString()))
                .thenReturn(Option.empty());
        for(Tuple2<String, String> paramValue : paramValues) {
            Mockito.when(request.getParameterO(paramValue.get1()))
                    .thenReturn(Option.of(paramValue.get2()));
        }
        for(String param : VALID_PARAMS.minus(paramValues.get1().unique())) {
            Mockito.when(request.getParameterO(param))
                    .thenReturn(Option.empty());
        }
        return request;
    }

    private static VideoInfoMpfsSourceMetaParser consSut() {
        return new VideoInfoMpfsSourceMetaParser();
    }

    @Test
    public void testIsYandexHostname() {
        Assert.isTrue(VideoInfoMpfsSourceMetaParser.isYandexHostname("iva7-2620bd90c191.qloud-c.YANDEX.NET"));
        Assert.isTrue(VideoInfoMpfsSourceMetaParser.isYandexHostname("ufo-trusty.dsd.yandex.net"));
    }

    @Test
    public void testNotIsYandexHostname() {
        Assert.isFalse(VideoInfoMpfsSourceMetaParser.isYandexHostname("www.nba.com"));
    }
}
