package ru.yandex.chemodan.app.videostreaming;

import java.io.IOException;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.cache.MpfsSegmentMeta;
import ru.yandex.chemodan.util.http.HttpException;
import ru.yandex.chemodan.videostreaming.framework.util.AsyncCacheableInputStreamSource;
import ru.yandex.chemodan.videostreaming.framework.util.CommonThreadPoolHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class VideoStreamingInternalClient {
    private static final Logger logger = LoggerFactory.getLogger(VideoStreamingInternalClient.class);

    private final int port;

    private final HttpClient httpClient;

    private final DiskStreamingUrlBuilder urlBuilder;

    public VideoStreamingInternalClient(HttpClient httpClient, DiskStreamingUrlBuilder urlBuilder, int port) {
        this.httpClient = httpClient;
        this.port = port;
        this.urlBuilder = urlBuilder;
    }

    public Option<InputStreamSource> getPrefetchedSegment(String host, MpfsSegmentMeta segmentMeta) {
        HttpResponse response;
        try {
            response = httpClient.execute(
                    new HttpHost(host, port),
                    new HttpGet(urlBuilder.getLocalSegmentPrefetchUrl(segmentMeta))
            );
        } catch (IOException e) {
            logger.warn("Error while connecting to sibling backend to access prefetching segment", e);
            throw ExceptionUtils.translate(e);
        }

        int statusCode = response.getStatusLine().getStatusCode();
        if (statusCode != HttpStatus.SC_200_OK) {
            if (statusCode != HttpStatus.SC_404_NOT_FOUND) {
                logger.error("Error while trying access prefetching segment: status code = {}", statusCode);
                throw new HttpException(statusCode);
            }

            EntityUtils.consumeQuietly(response.getEntity());
            logger.info("Prefetched segment is missing at remote host = {}", statusCode);
            return Option.empty();
        }

        AsyncCacheableInputStreamSource iss = new AsyncCacheableInputStreamSource();
        CommonThreadPoolHolder.runAsync(
                () -> {
                    try {
                        iss.receiveFrom(response.getEntity().getContent());
                    } catch (IOException | RuntimeException e) {
                        throw ExceptionUtils.translate(e);
                    } finally {
                        EntityUtils.consumeQuietly(response.getEntity());
                    }
                });
        return Option.of(iss);
    }
}
