package ru.yandex.chemodan.app.videostreaming.admin.mediainfo;

import java.net.URI;

import lombok.AllArgsConstructor;
import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpeg;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsFFmpegSourceProvider;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.EmptyContentPojo;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@ActionContainer
public class MediaInfoAdminPage {
    private static final String URL_PREFIX = "/streaming-media";

    private final UrlHelper urlHelper;

    private final MpfsSourceMetaParser mpfsSourceMetaParser;

    private final FFmpeg ffmpeg;

    private final HlsFFmpegSourceProvider<MpfsSourceMeta> sourceProvider;

    public MediaInfoAdminPage(UrlHelper urlHelper, MpfsSourceMetaParser mpfsSourceMetaParser, FFmpeg ffmpeg,
            HlsFFmpegSourceProvider<MpfsSourceMeta> sourceProvider)
    {
        this.urlHelper = urlHelper;
        this.mpfsSourceMetaParser = mpfsSourceMetaParser;
        this.ffmpeg = ffmpeg;
        this.sourceProvider = sourceProvider;
    }

    @ZAction(defaultAction = true)
    @Path(URL_PREFIX)
    public EmptyContentPojo index() {
        return new EmptyContentPojo();
    }

    @Path(URL_PREFIX + "/find")
    public EmptyContentPojo find(
            @RequestParam("url") String url,
            @RequestParam("stid") String stid)
    {
        stid = parseUrlMetaO(url)
                .map(MpfsSourceMeta::getStidStr)
                .getOrElse(stid);
        throw new ZRedirectException(
                new UriBuilder("/z" + URL_PREFIX + "/info")
                        .addParam("url", url)
                        .addParam("stid", stid)
                        .build()
                        .toASCIIString()
        );
    }

    @ZAction
    @Path(URL_PREFIX + "/info")
    public InfoPojo info(@RequestParam("url") String url, @RequestParam("stid") String stid) {
        return new InfoPojo(urlHelper.getUrlSet(stid), parseUrlMetaO(url));
    }

    private Option<MpfsSourceMeta> parseUrlMetaO(String url) {
        return StringUtils.notBlankO(url)
                .map(this::parseUrlMeta);
    }

    private MpfsSourceMeta parseUrlMeta(String url) {
        return mpfsSourceMetaParser.parse(URI.create(url).getPath(), false);
    }

    @ZAction
    @Path(URL_PREFIX + "/ffprobe")
    public FFprobePojo ffprobe(@RequestParam("stid") String stid) {
        return new FFprobePojo(
                ffmpeg.runFFprobeExecResult(
                        sourceProvider.get(
                                new MpfsSourceMeta(MulcaId.fromSerializedString(stid))
                        )
                )
        );
    }

    @BenderBindAllFields
    @XmlRootElement(name = "content")
    private static final class InfoPojo {
        final String pleerWithCacheUrl;
        final String pleerWithoutCacheUrl;
        final String playlistWithCacheUrl;
        final String playlistWithoutCacheUrl;
        final Option<UrlMeta> urlMeta;

        InfoPojo(UrlHelper.UrlSet urlSet, Option<MpfsSourceMeta> urlMeta) {
            this.pleerWithCacheUrl = urlSet.getPleer().toString();
            this.pleerWithoutCacheUrl = urlSet.getPleerNoCache().toString();
            this.playlistWithCacheUrl = urlSet.getBestQualityPlaylist().toString();
            this.playlistWithoutCacheUrl = urlSet.getBestQualityPlaylistNoCache().toString();
            this.urlMeta = urlMeta.map(UrlMeta::new);
        }
    }

    @BenderBindAllFields
    private static class UrlMeta {
        final String mulcaId;
        final Option<Long> owner;
        final Option<Long> consumer;
        final Option<String> userIp;
        final Option<String> argFiller;
        final Option<String> ffmpegVersion;

        UrlMeta(MpfsSourceMeta meta) {
            this.mulcaId = meta.getMulcaId().toSerializedString();
            this.owner = meta.getOwnerUid().map(AbstractPassportUid::getUid);
            this.consumer = meta.getConsumerUid().map(AbstractPassportUid::getUid);
            this.userIp = meta.getUserIp().map(IpAddress::format);
            this.argFiller = meta.getArgFiller();
            this.ffmpegVersion = meta.getFfVer();
        }
    }

    @BenderBindAllFields
    @XmlRootElement(name = "content")
    @Value
    @AllArgsConstructor
    private static class FFprobePojo {
        String stdout;

        String stderr;

        public FFprobePojo(ExecResult execResult) {
            this(execResult.getStdout(), execResult.getStderr());
        }
    }
}
