package ru.yandex.chemodan.app.videostreaming.cache;

import org.apache.http.client.HttpClient;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.inside.admin.conductor.Conductor;
import ru.yandex.inside.admin.conductor.ConductorContextConfiguration;
import ru.yandex.inside.admin.conductor.ConductorHosts;
import ru.yandex.inside.admin.conductor.GroupOrHost;
import ru.yandex.inside.mds.Mds;
import ru.yandex.inside.mds.MdsHosts;
import ru.yandex.inside.mds.MdsInternalDcAwarePingingProxies;
import ru.yandex.inside.mds.MdsInternalProxies;
import ru.yandex.inside.mds.MdsInternalStaticProxies;
import ru.yandex.inside.mds.MdsMetrics;
import ru.yandex.inside.mds.MdsNamespace;
import ru.yandex.inside.mds.MdsUrlSignParams;
import ru.yandex.misc.ip.HostPort;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import(ConductorContextConfiguration.class)
public class MdsConfig {
    private static final Duration DEFAULT_LINK_TTL = Duration.standardHours(4);

    @Value("${mds.read.host-port}")
    private HostPort readHostPort;

    @Value("${mds.write.host-port}")
    private HostPort writeHostPort;

    @Value("${mds.upload.group-or-host}")
    private GroupOrHost mdsUploadGroupOrHost;

    @Value("${mds.upload.port}")
    private int uploadPort;

    @Value("${mds.namespace}")
    private String namespace;

    @Value("${mds.namespace.auth-header}")
    private String authHeader;

    @Value("${mds.namespace.storage-sign-token}")
    private String storageSignToken;

    @Autowired
    @SuppressWarnings("SpringAutowiredFieldsWarningInspection")
    private Conductor conductor;

    @Bean
    public Mds mds() {
        HttpClient httpClient = mdsHttpClientConfigurator().configure();
        return new Mds(consHosts(), consNamespace(), httpClient, MdsMetrics.EMPTY);
    }

    private MdsHosts consHosts() {
        return new MdsHosts(readHostPort, writeHostPort, mdsUploadProxies());
    }

    @Bean
    public MdsInternalProxies mdsUploadProxies() {
        if (mdsUploadGroupOrHost.isHost()) {
            return new MdsInternalStaticProxies(Cf.list(new HostPort(mdsUploadGroupOrHost.getName(), uploadPort)));
        } else {
            ConductorHosts hosts = new ConductorHosts();
            hosts.setConductorGroupName(mdsUploadGroupOrHost.getName());
            hosts.setDefaultHostsCsv(conductor.getGroupHosts(mdsUploadGroupOrHost.getName()).mkString(","));

            return new MdsInternalDcAwarePingingProxies(conductor, hosts, uploadPort, mdsPingHttpClientConfigurator().configure());
        }
    }

    @Bean
    @OverridableValuePrefix("mds_ping")
    public HttpClientConfigurator mdsPingHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    @OverridableValuePrefix("mds")
    public HttpClientConfigurator mdsHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    private MdsNamespace consNamespace() {
        return new MdsNamespace(
                namespace,
                authHeader,
                Option.of(new MdsUrlSignParams(storageSignToken, DEFAULT_LINK_TTL))
        );
    }
}
