package ru.yandex.chemodan.app.videostreaming.cache;

import java.util.UUID;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.app.videostreaming.VideoStreamingInternalClient;
import ru.yandex.chemodan.util.http.HttpException;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.CacheData;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.HlsSegmentCache;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.SegmentIndexes;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.util.CommonThreadPoolHolder;
import ru.yandex.chemodan.videostreaming.framework.util.IntervalRunner;
import ru.yandex.inside.mds.Mds;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mds.MdsPostResponse;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MdsSegmentCache implements HlsSegmentCache<MpfsSourceMeta> {
    private static final Logger logger = LoggerFactory.getLogger(MdsSegmentCache.class);

    private final SegmentCacheDao segmentCacheDao;

    private final Mds mds;

    private final VideoStreamingInternalClient internalClient;

    private final IntervalRunner lastAccessTimeUpdater;

    public MdsSegmentCache(SegmentCacheDao segmentCacheDao, Mds mds, VideoStreamingInternalClient internalClient,
                           IntervalRunner lastAccessTimeUpdater)
    {
        this.segmentCacheDao = segmentCacheDao;
        this.mds = mds;
        this.internalClient = internalClient;
        this.lastAccessTimeUpdater = lastAccessTimeUpdater;
    }

    @Override
    public CacheData getCacheData(HlsSegmentMeta<MpfsSourceMeta> segmentMeta) {
        return getCacheData(new MpfsSegmentMeta(segmentMeta));
    }

    private CacheData getCacheData(MpfsSegmentMeta segmentMeta) {
        Option<SegmentCacheMeta> cacheMetaO = getSegmentCacheMetaO(segmentMeta);
        lastAccessTimeUpdater.runAsyncIfAppropriate(
                cacheMetaO.map(SegmentCacheMeta::getLastAccessTime),
                () -> segmentCacheDao.updateLastAccessTime(segmentMeta)
        );
        return cacheMetaO.map(cacheMeta -> consCacheData(cacheMeta, segmentMeta))
                .getOrElse(CacheData.ABSENT);
    }

    protected Option<SegmentCacheMeta> getSegmentCacheMetaO(MpfsSegmentMeta segmentMeta) {
        return segmentCacheDao.getSegmentCacheMetaO(segmentMeta);
    }

    private CacheData consCacheData(SegmentCacheMeta cacheMeta, MpfsSegmentMeta segmentMeta) {
        return cacheMeta.getKeyO()
                .map(key -> CacheData.fromCache(mds.getInputStreamSourceViaRedirect(key, Duration.standardMinutes(10))))
                .orElse(() ->
                        cacheMeta.getTranscodingHostO()
                                .map(host -> fetchRemoteSegment(host, segmentMeta))
                )
                .getOrElse(CacheData.IN_PROGRESS);
    }

    private CacheData fetchRemoteSegment(String host, MpfsSegmentMeta segmentMeta) {
        try {
            return fetchRemoteSegmentUnsafe(host, segmentMeta);
        } catch (RuntimeException ex) {
            if (!isRetryable(ex)) {
                return CacheData.ABSENT;
            }

            try {
                return fetchRemoteSegmentUnsafe(host, segmentMeta);
            } catch (RuntimeException ex2) {
                return CacheData.ABSENT;
            }
        }
    }

    private static boolean isRetryable(RuntimeException ex) {
        return ex instanceof RuntimeIoException || (ex instanceof HttpException && ((HttpException) ex).is5xx());
    }

    private CacheData fetchRemoteSegmentUnsafe(String host, MpfsSegmentMeta segmentMeta) {
        return internalClient.getPrefetchedSegment(host, segmentMeta)
                .map(CacheData::fromPrefetch)
                .getOrElse(CacheData.ABSENT);
    }

    @Override
    public void put(HlsSegmentMeta<MpfsSourceMeta> segmentMeta, InputStreamSource iss) {
        put(new MpfsSegmentMeta(segmentMeta), iss);
    }

    private void put(MpfsSegmentMeta segmentMeta, InputStreamSource iss) {
        MdsPostResponse mdsResp = mds.upload(UUID.randomUUID().toString(), iss);
        putIfAbsentDeleteDataOtherwise(segmentMeta, mdsResp.getKey());
    }

    private void putIfAbsentDeleteDataOtherwise(MpfsSegmentMeta segmentMeta, MdsFileKey key) {
        boolean keyWasSet = segmentCacheDao.setKeyIfAbsent(segmentMeta, key);
        if (!keyWasSet) {
            deleteData(key);
        }
    }

    private void deleteData(MdsFileKey key) {
        mds.delete(key);
    }

    @Override
    public boolean prepare(HlsSegmentMeta<MpfsSourceMeta> segmentMeta) {
        return prepare(new MpfsSegmentMeta(segmentMeta));
    }

    private boolean prepare(MpfsSegmentMeta segmentMeta) {
        return segmentCacheDao.createOrUpdateEmpty(segmentMeta);
    }

    @Override
    public SegmentIndexes getCachedSegmentIndexes(MpfsSourceMeta srcMeta, HlsStreamQuality quality,
            MediaTime duration)
    {
        return new SegmentIndexes(
                MasterSlaveContextHolder.withPolicyIgnoreErrors(
                        MasterSlavePolicy.R_S,
                        () -> getCachedSegmentIndexList(srcMeta, quality, duration)
                ).getOrElse(Cf.list())
        );
    }

    private ListF<ListF<Integer>> getCachedSegmentIndexList(MpfsSourceMeta srcMeta, HlsStreamQuality quality,
            MediaTime duration)
    {
        return segmentCacheDao.getCachedSegmentIndexes(srcMeta, quality, duration);
    }

    @Override
    public void clearCacheForSource(MpfsSourceMeta sourceMeta) {
        ListF<MdsFileKey> mdsFileKeys = segmentCacheDao.deleteSegmentsWithMulcaId(sourceMeta.getMulcaId());
        CommonThreadPoolHolder.runAsync(() -> deleteData(mdsFileKeys));
    }

    private void deleteData(ListF<MdsFileKey> mdsFileKeys) {
        mdsFileKeys.forEach(this::deleteDataSafe);
    }

    private void deleteDataSafe(MdsFileKey mdsFileKey) {
        try {
            logger.info("Deleting segment cache {}", mdsFileKey);
            deleteData(mdsFileKey);
            logger.info("Deleted segment cache {}", mdsFileKey);
        } catch(RuntimeException ex) {
            logger.error("Error deleting segment cache {}", mdsFileKey);
        }
    }
}
