package ru.yandex.chemodan.app.videostreaming.cache;

import java.sql.ResultSet;
import java.sql.SQLException;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.videostreaming.framework.hls.DurationUtil;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.mds.MdsFileKey;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class SegmentCacheDao extends AbstractVideoStreamingDao {
    private static final DynamicProperty<Boolean> readFromMaster =
            DynamicProperty.cons("streaming-read-segment-cache-from-master", false);

    private final JdbcTemplate3 jdbcTemplate;

    private final SegmentCacheIdProvider cacheIdProvider;

    public SegmentCacheDao(JdbcTemplate3 jdbcTemplate, SegmentCacheIdProvider cacheIdProvider) {
        this.jdbcTemplate = jdbcTemplate;
        this.cacheIdProvider = cacheIdProvider;
    }

    @RunWithRandomTest
    public ListF<ListF<Integer>> getCachedSegmentIndexes(MpfsSourceMeta srcMeta, HlsStreamQuality quality,
            MediaTime duration)
    {
        return jdbcTemplate.query(
                "WITH indexes AS ("
                            + " SELECT index FROM segment_cache WHERE mulca_id = ? AND quality = ? AND duration = ?"
                            + " ORDER BY index"
                        + "),"
                        + "grouped_indexes AS ("
                            + "SELECT ROW_NUMBER() OVER (ORDER BY index) - index AS grp, index FROM indexes"
                        + ")"
                        + " SELECT MIN(index) AS from, MAX(index) AS to FROM grouped_indexes"
                        + " GROUP BY grp ORDER BY MIN(index)",
                (rs, rowNum) -> Cf.range(rs.getInt("from"), rs.getInt("to") + 1),
                srcMeta.getMulcaId().toSerializedString(),
                quality.toString(),
                DurationUtil.toSecondsWith2DecimalPlaces(duration)
        );
    }

    @RunWithRandomTest
    public Option<SegmentCacheMeta> getSegmentCacheMetaO(MpfsSegmentMeta segmentMeta) {
        if (readFromMaster.get() && MasterSlaveContextHolder.policy() != MasterSlavePolicy.R_SM) {
            return getSegmentCacheMetaRawO(segmentMeta);
        } else {
            Function0<Option<SegmentCacheMeta>> fetchF = () -> getSegmentCacheMetaRawO(segmentMeta);
            return MasterSlaveContextHolder.withPolicyIgnoreErrors(MasterSlavePolicy.R_S, fetchF)
                    .getOrElse(Option::empty)
                    .orElse(() -> MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_M, fetchF));
        }
    }

    private Option<SegmentCacheMeta> getSegmentCacheMetaRawO(MpfsSegmentMeta segmentMeta) {
        SqlCondition sqlCondition = consIdCondition(segmentMeta);
        return jdbcTemplate.queryForOption(
                "SELECT mds_key, transcoding_host, last_access_time FROM segment_cache" + sqlCondition.whereSql(),
                (rs, rowNum) -> consCacheMeta(rs),
                sqlCondition.args()
        );
    }

    @RunWithRandomTest
    public void updateLastAccessTime(MpfsSegmentMeta segmentMeta) {
        SqlCondition sqlCondition = consIdCondition(segmentMeta)
                .and(SqlCondition.column("mds_key").isNotNull());
        jdbcTemplate.update(
                "UPDATE segment_cache SET last_access_time = ?" + sqlCondition.whereSql(),
                Instant.now(),
                sqlCondition.args()
        );
    }

    @RunWithRandomTest
    public boolean createOrUpdateEmpty(MpfsSegmentMeta segmentMeta) {
        return createOrUpdateEmpty(getCacheId(segmentMeta));
    }

    private boolean createOrUpdateEmpty(SegmentCacheId segmentMeta) {
        int count = jdbcTemplate.update(
                "INSERT into segment_cache (mulca_id, quality, index, duration, last_access_time, transcoding_host)"
                        + " VALUES (?, ?, ?, ?, ?, ?)"
                        + " ON CONFLICT (mulca_id, quality, index, duration) DO UPDATE"
                        + " SET last_access_time = EXCLUDED.last_access_time"
                            + ", transcoding_host = EXCLUDED.transcoding_host"
                        + " WHERE segment_cache.mds_key is NULL",
                segmentMeta.getMulcaIdWithTag(),
                segmentMeta.getQualityStr(),
                segmentMeta.getIndex(),
                segmentMeta.getDurationInSeconds(),
                Instant.now(),
                HostnameUtils.localHostname()
        );
        return count != 0;
    }

    @RunWithRandomTest
    public boolean setKeyIfAbsent(MpfsSegmentMeta segmentMeta, MdsFileKey mdsFileKey) {
        SqlCondition sqlCondition = consIdCondition(segmentMeta)
                .and(SqlCondition.column("mds_key").isNull());
        int count = jdbcTemplate.update(
                "UPDATE segment_cache SET mds_key = ?, last_access_time = ?, transcoding_host = NULL" +
                        sqlCondition.whereSql(),
                mdsFileKey.serialize(),
                Instant.now(),
                sqlCondition.args()
        );
        return count != 0;
    }

    @RunWithRandomTest
    public ListF<MdsFileKey> deleteSegmentsWithMulcaId(MulcaId mulcaId) {
        return jdbcTemplate.query(
                "DELETE FROM segment_cache WHERE mulca_id = ? RETURNING mds_key",
                (rs, rowNum) -> Option.ofNullable(rs.getString("mds_key")),
                mulcaId.toSerializedString()
        ).filterMap(key -> key.map(MdsFileKey::parse));
    }

    private static SegmentCacheMeta consCacheMeta(ResultSet rs) throws SQLException {
        return new SegmentCacheMeta(
                Option.ofNullable(rs.getString("mds_key"))
                        .map(MdsFileKey::parse),
                Option.ofNullable(rs.getString("transcoding_host")),
                new Instant(rs.getTimestamp("last_access_time"))
        );
    }

    private SqlCondition consIdCondition(MpfsSegmentMeta segmentMeta) {
        return consIdCondition(getCacheId(segmentMeta));
    }

    private SegmentCacheId getCacheId(MpfsSegmentMeta segmentMeta) {
        return cacheIdProvider.getCacheId(segmentMeta);
    }

    private static SqlCondition consIdCondition(SegmentCacheId cacheId) {
        return SqlCondition.column("mulca_id").eq(cacheId.getMulcaIdWithTag())
                .and(SqlCondition.column("quality").eq(cacheId.getQualityStr()))
                .and(SqlCondition.column("index").eq(cacheId.getIndex()))
                .and(SqlCondition.column("duration").eq(cacheId.getDurationInSeconds()));
    }
}
