package ru.yandex.chemodan.app.videostreaming.cache;

import org.joda.time.Instant;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.app.videostreaming.overrides.RequestFilter;
import ru.yandex.chemodan.app.videostreaming.overrides.TranscodeOverride;
import ru.yandex.chemodan.app.videostreaming.overrides.TranscodeOverridesRegistry;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsResource;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentRegion;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.TranscodeParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.TestHelper;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class SegmentCacheIdProviderTest {
    private static final Instant EXPIRE_AT = new Instant(123000);

    private static final String FFMPEG_VERSION = FFmpegParams.DEFAULT.getFfmpegVersion();

    @Test
    public void tagContainsFFmpegVersionForDefaultArgFiller() {
        assertCacheIdHasTag(Option.of("$" + FFMPEG_VERSION + "$"), "default");
    }

    @Test
    public void tagMatchesArgFiller() {
        assertCacheIdHasTag(Option.of("$some-name:" + FFMPEG_VERSION + "$"), "some-name");
    }

    @Test
    public void expireAtIsAddedWhenCacheDisabled() {
        assertCacheIdHasTag(Option.of("$name:" + FFMPEG_VERSION + ":123$"), "name", Option.of(
                new TranscodeOverride("override", new RequestFilter(Cf.list()),
                        TranscodeParams.DEFAULT.toBuilder()
                                .disableCache(true)
                                .build()
                )
        ));
    }

    private void assertCacheIdHasTag(Option<String> expectedTagO, String argFillerName) {
        assertCacheIdHasTag(expectedTagO, argFillerName, Option.empty());
    }

    private void assertCacheIdHasTag(Option<String> expectedTagO, String argFillerName,
            Option<TranscodeOverride> overrideO)
    {
        HlsResource.Stream stream = TestHelper.buildHlsResource()
                .getStream(HlsStreamQuality._720P);
        HlsSegmentRegion segment = new HlsSegmentRegion(1, MediaTime.seconds(5), stream);
        MulcaId mulcaId = MulcaId.valueOf("123", "456");
        MpfsSegmentMeta mpfsSegmentMeta = new MpfsSegmentMeta(
                stream,
                segment,
                new MpfsSourceMeta(
                        mulcaId,
                        Option.empty(),
                        Option.empty(),
                        false,
                        Option.empty(),
                        Option.of(EXPIRE_AT),
                        Option.empty(),
                        Option.empty()
                )
        );
        TranscodeOverridesRegistry overridesRegistry = Mockito.mock(TranscodeOverridesRegistry.class);
        Mockito.when(overridesRegistry.getOverrideO(Mockito.any()))
                .thenReturn(overrideO);
        SegmentCacheIdProvider sut = new SegmentCacheIdProvider(
                overridesRegistry,
                FFmpegParams.DEFAULT,
                HlsFFmpegCommandParams.DEFAULT.toBuilder()
                        .argFillerName(argFillerName)
                        .build()
        );
        Assert.equals(
                new SegmentCacheId(mulcaId, HlsStreamQuality._720P, segment)
                        .withTagO(expectedTagO),
                sut.getCacheId(mpfsSegmentMeta)
        );
    }
}
