package ru.yandex.chemodan.app.videostreaming.cache;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import lombok.RequiredArgsConstructor;
import lombok.val;

import ru.yandex.bolts.collection.Option;

@RequiredArgsConstructor
public class SoftMap<K, V> {

    private final ReferenceQueue<V> queue = new ReferenceQueue<>();
    private final Map<K, Ref> map = new ConcurrentHashMap<>();
    private final long ttl;

    public Option<V> get(K key) {
        return Option.ofNullable(map.get(key)).flatMapO(ref -> Option.ofNullable(ref.get()));
    }

    public void put(K key, V value) {
        map.put(key, new Ref(key, value));
    }

    private class Ref extends SoftReference<V> {

        final K key;
        long access = System.currentTimeMillis();

        Ref(K key, V value) {
            super(value, queue);
            this.key = key;
            for (Reference<?> ref = queue.poll(); ref != null; ref = queue.poll()) {
                ref.clear();
            }
        }

        @Override
        public synchronized V get() {
            val now = System.currentTimeMillis();
            if (ttl > 0 && now > access + ttl) {
                clear();
            } else {
                access = now;
            }
            return super.get();
        }

        @Override
        public void clear() {
            super.clear();
            map.remove(key, this);
        }

    }

}
