package ru.yandex.chemodan.app.videostreaming.cache;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class VideoInfoCacheDao extends AbstractVideoStreamingDao {
    private static final int CURRENT_VERSION = 2;

    private final JdbcTemplate3 jdbcTemplate;

    public VideoInfoCacheDao(JdbcTemplate3 jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    public Option<Tuple2<MediaInfo, Instant>> getVideoInfoWithLastAccessTimeO(MulcaId mulcaId) {
        MasterSlavePolicy masterSlavePolicy = MasterSlaveContextHolder.policyO().getOrElse(MasterSlavePolicy.R_SM);
        return MasterSlaveContextHolder.withPolicy(masterSlavePolicy,
                () -> jdbcTemplate.queryForOption(
                        "SELECT ffprobe_output, last_access_time FROM ffprobe_cache WHERE mulca_id = ?"
                                + " AND version >= ?",
                        (rs, rowNum) -> new Tuple2<>(
                                MediaInfo.parse(rs.getString("ffprobe_output")),
                                new Instant(rs.getTimestamp("last_access_time"))
                        ),
                        mulcaId.toSerializedString(),
                        CURRENT_VERSION
                )
        );
    }

    @RunWithRandomTest
    public void updateLastAccessTime(MulcaId mulcaId) {
        jdbcTemplate.update("UPDATE ffprobe_cache SET last_access_time = ? WHERE mulca_id = ?",
                Instant.now(),
                mulcaId.toSerializedString()
        );
    }

    @RunWithRandomTest
    public void save(MulcaId mulcaId, MediaInfo mediaInfo) {
        jdbcTemplate.update(
                "INSERT into ffprobe_cache (mulca_id, ffprobe_output, version, last_access_time) VALUES (?, ?, ?, ?)"
                        + " ON CONFLICT (mulca_id) DO UPDATE"
                        + " SET last_access_time = EXCLUDED.last_access_time"
                            + ", ffprobe_output = EXCLUDED.ffprobe_output"
                            + ", version = EXCLUDED.version",
                mulcaId.toSerializedString(),
                mediaInfo.getJson(),
                CURRENT_VERSION,
                Instant.now()
        );
    }

    @RunWithRandomTest
    public void deleteVideoInfoWithMulcaId(MulcaId mulcaId) {
        jdbcTemplate.update("DELETE FROM ffprobe_cache WHERE mulca_id = ?", mulcaId.toSerializedString());
    }
}
