package ru.yandex.chemodan.app.videostreaming.config;

import java.net.InetAddress;
import java.util.concurrent.TimeUnit;

import javax.cache.expiry.ModifiedExpiryPolicy;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteSystemProperties;
import org.apache.ignite.cache.affinity.rendezvous.RendezvousAffinityFunction;
import org.apache.ignite.cache.eviction.lru.LruEvictionPolicyFactory;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.configuration.IgniteConfiguration;
import org.apache.ignite.failure.StopNodeFailureHandler;
import org.apache.ignite.spi.discovery.DiscoverySpi;
import org.apache.ignite.spi.discovery.tcp.TcpDiscoverySpi;
import org.apache.ignite.spi.discovery.tcp.ipfinder.TcpDiscoveryIpFinder;
import org.apache.ignite.spi.discovery.tcp.ipfinder.zk.TcpDiscoveryZookeeperIpFinder;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Profile;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.boot.ChemodanCommonContextConfiguration;
import ru.yandex.chemodan.videostreaming.framework.config.BannedIpConfig;
import ru.yandex.chemodan.videostreaming.framework.ignite.ExternalAddressResolver;
import ru.yandex.chemodan.videostreaming.framework.ignite.IcebergIgniteLogger;
import ru.yandex.chemodan.videostreaming.framework.ignite.IgniteBannedIpUpdater;
import ru.yandex.chemodan.videostreaming.framework.ignite.IgniteSpringBean;
import ru.yandex.chemodan.videostreaming.framework.web.BannedIpRegistry;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.alive2.location.LocationResolverConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.commune.zk2.ZkConfiguration;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.Version;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({ChemodanCommonContextConfiguration.class, LocationResolverConfiguration.class, BannedIpConfig.class})
@Profile("ignite")
public class IgniteConfig {
    private static final Logger logger = LoggerFactory.getLogger(Ignite.class);

    private static final Duration DEFAULT_ZK_CONNECTION_TIMEOUT = Duration.standardSeconds(15);

    @Autowired
    @Qualifier("zkRoot")
    private ZkPath zkRoot;

    @Autowired
    private AppName appName;

    @Autowired
    private Version version;

    @Autowired
    public ZkConfiguration zkConfiguration;

    @Autowired
    private LocationResolver locationResolver;

    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    @Autowired
    private BannedIpRegistry bannedIpRegistry;

    @Value("${ignite.log.quiet}")
    public boolean quiet;

    @Value("${ignite.zookeeper.group}")
    public String igniteZookeeperGroup;

    @Value("${streaming.ip-mismatch.cache.max-size}")
    private int ipMismatchCacheMaxSize;

    @Value("${streaming.ip-mismatch.cache.ttl}")
    private Duration ipMismatchCacheTtl;

    @Bean
    public IgniteSpringBean igniteSpringBean() {
        // disable GridUpdateNotifier, which periodically sends stats to ignite.apache.org
        // somebody from Apache Ignite team called this "notification about new version availability" :)
        System.setProperty(IgniteSystemProperties.IGNITE_UPDATE_NOTIFIER, "false");

        dynamicPropertyManager.addStaticFields(IgniteSpringBean.class);

        return new IgniteSpringBean(this::getIgniteConfiguration);
    }

    private IgniteConfiguration getIgniteConfiguration() {
        IgniteConfiguration igniteConfig = new IgniteConfiguration();
        igniteConfig.setDiscoverySpi(getDiscoverySpi());
        igniteConfig.setFailureHandler(new StopNodeFailureHandler());
        igniteConfig.setGridLogger(new IcebergIgniteLogger(quiet));
        return igniteConfig;
    }

    private DiscoverySpi getDiscoverySpi() {
        TcpDiscoverySpi discoverySpi = new TcpDiscoverySpi();
        discoverySpi.setIpFinder(consIpFinder());
        discoverySpi.setAddressResolver(consAddressResolverOrNull());
        return discoverySpi;
    }

    private ExternalAddressResolver consAddressResolverOrNull() {
        ExternalAddressResolver addressResolver = new ExternalAddressResolver();
        if (addressResolver.isEmpty()) {
            logger.warn("No addresses found for Ignite cluster discovery - using default resolver");
            return null;
        }

        logger.info("Using following addresses for Ignite cluster discovery: {}",
                addressResolver.getAddresses().map(InetAddress::toString));

        return addressResolver;
    }

    private TcpDiscoveryIpFinder consIpFinder() {
        TcpDiscoveryZookeeperIpFinder ipFinder = new TcpDiscoveryZookeeperIpFinder();
        ipFinder.setCurator(consCurator());
        ipFinder.setBasePath(curatorZkPath().getPath());
        ipFinder.setServiceName("apacheIgnite-tcpDiscovery");

        logger.info("Using zk IP finder with base path = {} and service name = {} for Ignite discovery",
                ipFinder.getBasePath(), ipFinder.getServiceName());

        return ipFinder;
    }

    private CuratorFramework consCurator() {
        return CuratorFrameworkFactory.newClient(
                zkConfiguration.getConnectionUrl(),
                zkConfiguration.getTimeout(),
                (int) DEFAULT_ZK_CONNECTION_TIMEOUT.getMillis(),
                new ExponentialBackoffRetry(1000, 10)
        );
    }

    private ZkPath curatorZkPath() {
        ListF<String> path = Cf.list(appName.appName(), "curator", version.getProjectVersion())
                .plus(igniteZookeeperGroup)
                .plus(locationResolver.resolveLocation().dcName);
        return zkRoot.child(StringUtils.join(path, "/"));
    }

    @Bean
    public IgniteBannedIpUpdater igniteBannedIpUpdater() {
        dynamicPropertyManager.addStaticFields(IgniteBannedIpUpdater.class);
        return new IgniteBannedIpUpdater(igniteSpringBean(), ipMismatchCacheConfig(), bannedIpRegistry);
    }

    @Bean
    protected CacheConfiguration<IpAddress, IgniteBannedIpUpdater.IpAddresses> ipMismatchCacheConfig() {
        CacheConfiguration<IpAddress, IgniteBannedIpUpdater.IpAddresses> cacheConfig =
                new CacheConfiguration<>("ipMismatch");
        cacheConfig.setAffinity(new RendezvousAffinityFunction());
        cacheConfig.setOnheapCacheEnabled(true);
        cacheConfig.setEvictionPolicyFactory(new LruEvictionPolicyFactory<>(ipMismatchCacheMaxSize));
        cacheConfig.setExpiryPolicyFactory(ModifiedExpiryPolicy.factoryOf(getIpMismatchCacheTtl()));
        return cacheConfig;
    }

    private javax.cache.expiry.Duration getIpMismatchCacheTtl() {
        return new javax.cache.expiry.Duration(TimeUnit.MILLISECONDS, ipMismatchCacheTtl.getMillis());
    }
}
