package ru.yandex.chemodan.app.videostreaming.config;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.app.videostreaming.VideoInfoMpfsSourceMetaParser;
import ru.yandex.chemodan.app.videostreaming.admin.StreamingAdminConfig;
import ru.yandex.chemodan.boot.ChemodanCommonContextConfiguration;
import ru.yandex.chemodan.boot.admin.ChemodanAdminDaemonContextConfiguration;
import ru.yandex.chemodan.util.AvailablePortFinder;
import ru.yandex.chemodan.util.spring.MapProperties;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.UpstreamIdProvider;
import ru.yandex.chemodan.videostreaming.framework.config.FFmpegReceiverOutputConfig;
import ru.yandex.chemodan.videostreaming.framework.config.FFmpegVersions;
import ru.yandex.chemodan.videostreaming.framework.config.HlsCachingSourceProxyConfig;
import ru.yandex.chemodan.videostreaming.framework.config.HlsCachingSourceProxyParams;
import ru.yandex.chemodan.videostreaming.framework.config.VideoInfoQualifier;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFPaths;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandParams;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.HlsSegmentPrefetchTrigger;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaConverter;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.StreamingResourceParams;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.VideoInfoParams;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.util.SingleWarJettyUtil;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({
        HlsCachingSourceProxyConfig.class,
        FFmpegReceiverOutputConfig.class,
        MulcaUpstreamConfig.class,
        VideoStreamingCoreConfig.ProxyPortConfig.class,
        ChemodanAdminDaemonContextConfiguration.class,
        ChemodanCommonContextConfiguration.class,
        StreamingAdminConfig.class,
})
public class VideoStreamingCoreConfig {
    private final HlsCachingSourceProxyConfig sourceConfig;

    private final FFmpegReceiverOutputConfig ffmpegReceiverOutputConfig;

    private final ProxyPortConfig proxyPortConfig;

    private final DynamicPropertyManager dynamicPropertyManager;

    @Autowired
    public VideoStreamingCoreConfig(
            HlsCachingSourceProxyConfig sourceConfig,
            FFmpegReceiverOutputConfig ffmpegReceiverOutputConfig,
            ProxyPortConfig proxyPortConfig,
            DynamicPropertyManager dynamicPropertyManager)
    {
        this.sourceConfig = sourceConfig;
        this.ffmpegReceiverOutputConfig = ffmpegReceiverOutputConfig;
        this.proxyPortConfig = proxyPortConfig;
        this.dynamicPropertyManager = dynamicPropertyManager;
    }

    @Configuration
    public static class InnerConfig {
        @Bean
        public UpstreamIdProvider<MpfsSourceMeta, MulcaId> upstreamIdProvider() {
            return MpfsSourceMeta::getMulcaId;
        }
    }

    @Bean
    @VideoInfoQualifier
    public SourceMetaParser<MpfsSourceMeta> videoInfoSourceMetaParser() {
        dynamicPropertyManager.addStaticFields(VideoInfoMpfsSourceMetaParser.class);
        return new VideoInfoMpfsSourceMetaParser();
    }

    @Bean
    public SourceMetaConverter<MpfsSourceMeta> sourceMetaConverter() {
        return new SourceMetaConverter<>("stid", MpfsSourceMeta::getStidStr);
    }

    @Bean
    public HlsSegmentPrefetchTrigger segmentPrefetchTrigger(
            @Value("${segment-prefetch.min-elapsed}") Duration minElapsed)
    {
        return stats -> stats.elapsed.isLongerThan(minElapsed) ? ((int) Math.ceil(1 / stats.speed)) - 1 : 0;
    }

    @Bean
    public SingleWarJetty internalJetty(
            @Value("${internal-jetty.http.max-threads}")
                    int maxThreads,
            @Value("${internal-jetty.http.min-threads}")
                    int minThreads,
            @Value("${internal-jetty.http.max-queue-length}")
                    int maxQueueLength,
            @Value("${caching-src-proxy.http.url-prefix}")
                    String proxyUrlPrefix)
    {
        SingleWarJetty jetty = SingleWarJettyUtil.consSingleWarJetty(
                proxyPortConfig.getInternalJettyPort(), maxThreads, minThreads, maxQueueLength);
        jetty.addServletMapping("/" + proxyUrlPrefix + "/*", sourceConfig.storageProxyServlet());
        jetty.addServletMapping("/ffmpeg/*", ffmpegReceiverOutputConfig.ffmpegReceiverServlet());
        return jetty;
    }

    @Configuration
    @Import({ProxyPortConfig.class})
    @SuppressWarnings("unused")
    static class HlsParamsConfig {
        @Value("${ffmpeg.in-threads}")
        private int ffmpegInputThreads;

        @Value("${ffmpeg.out-threads}")
        private int ffmpegOutputThreads;

        @Value("${ffmpeg.use-gapless}")
        private boolean useGapless;

        @Value("${ffmpeg.recv-buffer-size}")
        private DataSize recvBufferSize;

        @Value("${ffmpeg.max-process-count}")
        private int maxProcCount;

        @Value("${ffmpeg.transcoding-timeout}")
        private Duration ffmpegTranscodingTimeout;

        @Value("${ffmpeg.encoding-start-timeout}")
        private Duration ffmpegEncodingStartTimeout;

        @Value("${ffmpeg.min-left-over-duration}")
        private Duration minLeftOverDuration;

        @Value("${caching-src-proxy.http.url-prefix}")
        private String proxyUrlPrefix;

        @Value("${caching-src-proxy.cache.size}")
        private int cacheSize;

        @Value("${caching-src-proxy.cache.data-size}")
        private DataSize cacheDataSize;

        @Value("${caching-src-proxy.upstream-retries}")
        private int upstreamRetryCount;

        @Value("${hls.segment-duration}")
        private Duration segmentDuration;

        @Value("${hls.prefetch.initial-count}")
        private int initialPrefetchCount;

        @Value("${hls.prefetch.max-count}")
        private int maxPrefetchCount;

        @Value("${hls.max-cache-wait}")
        private Duration maxCacheWait;

        @Value("${hls.float-segment-durations-enabled}")
        private boolean floatSegmentDurationsEnabled;

        @Value("${streaming.enable-full-hd}")
        private boolean enableFullHd;

        private final ProxyPortConfig proxyPortConfig;

        @Autowired
        public HlsParamsConfig(ProxyPortConfig proxyPortConfig) {
            this.proxyPortConfig = proxyPortConfig;
        }

        @Bean
        protected MapProperties ffmpegBinariesProperties() {
            return new MapProperties("ffmpeg.binaries");
        }

        @Bean
        public FFmpegVersions ffmpegVersions() {
            MapF<String, String> properties = ffmpegBinariesProperties()
                    .get()
                    .filterValues(StringUtils::isNotBlank);
            Pattern2 versionPattern = Pattern2.compile("([^.]+)\\.");
            MapF<String, FFPaths> paths = properties.keys()
                    .filterMap(versionPattern::findFirstGroup)
                    .unique()
                    .toMapMappingToValue(version -> new FFPaths(
                            properties.getTs(version + ".ffmpeg.path"),
                            properties.getTs(version + ".ffprobe.path")
                    ));
            return new FFmpegVersions(paths);
        }

        @Bean
        public HlsParams hlsParams() {
            return new HlsParams(
                    new StreamingParams(MediaTime.fromDuration(segmentDuration),
                            initialPrefetchCount, maxPrefetchCount, maxCacheWait, floatSegmentDurationsEnabled),
                    new VideoInfoParams(),
                    new FFmpegParams(ffmpegTranscodingTimeout, ffmpegEncodingStartTimeout, minLeftOverDuration,
                            maxProcCount),
                    new HlsFFmpegCommandParams(ffmpegInputThreads, ffmpegOutputThreads, useGapless, recvBufferSize),
                    new StreamingResourceParams(enableFullHd)
            );
        }

        @Bean
        public HlsCachingSourceProxyParams hlsProxyParams() {
            return new HlsCachingSourceProxyParams(
                    proxyUrlPrefix,
                    proxyPortConfig.getInternalJettyPort(),
                    cacheSize,
                    cacheDataSize,
                    upstreamRetryCount
            );
        }
    }

    @Configuration
    static class ProxyPortConfig {
        @Value("${internal-jetty.http.port-range}")
        private String httpPortRange;

        public int getInternalJettyPort() {
            return AvailablePortFinder.getNextAvailable(parseRangeAsTuple2(httpPortRange));
        }
    }

    private static Tuple2<Integer, Integer> parseRangeAsTuple2(String str) {
        String[] array = str.split("-");
        return new Tuple2<>(Integer.parseInt(array[0]), Integer.parseInt(array[1]));
    }
}
