package ru.yandex.chemodan.app.videostreaming.config;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.videostreaming.DiskStreamingSegmentPrefetch;
import ru.yandex.chemodan.app.videostreaming.DiskStreamingUrlBuilder;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMetaTransformer;
import ru.yandex.chemodan.app.videostreaming.TransitionalValue;
import ru.yandex.chemodan.app.videostreaming.VideoStreamingInternalClient;
import ru.yandex.chemodan.app.videostreaming.cache.DiskStreamingVideoInfoCache;
import ru.yandex.chemodan.app.videostreaming.cache.DiskStreamingVideoInfoCacheWithMemoryCache;
import ru.yandex.chemodan.app.videostreaming.cache.MdsConfig;
import ru.yandex.chemodan.app.videostreaming.cache.MdsSegmentCache;
import ru.yandex.chemodan.app.videostreaming.cache.MdsSegmentCacheWithMemoryCache;
import ru.yandex.chemodan.app.videostreaming.cache.SegmentCacheIdProvider;
import ru.yandex.chemodan.app.videostreaming.overrides.TranscodeOverrideConfig;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.util.encrypt.OpenSslAes256CbcCrypter;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.jdbc.DataSourceProperties;
import ru.yandex.chemodan.util.jdbc.JdbcDatabaseConfiguratorContextConfiguration;
import ru.yandex.chemodan.videostreaming.framework.config.HlsConfig;
import ru.yandex.chemodan.videostreaming.framework.hls.ForceFFmpeg3ForFormats;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaTransformer;
import ru.yandex.chemodan.videostreaming.framework.util.IntervalRunner;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({
        JdbcDatabaseConfiguratorContextConfiguration.class,
        HlsConfig.class,
        MdsConfig.class,
        VideoStreamingCoreConfig.class,
        VideoStreamingHlsConfig.DaoConfig.class,
        VideoStreamingHlsConfig.SegmentCacheIdProviderConfig.class,
        VideoStreamingDaoConfig.class,
        IgniteConfig.class
})
@RequiredArgsConstructor
public class VideoStreamingHlsConfig {
    private final MdsConfig mdsConfig;

    private final VideoStreamingDaoConfig daoConfig;

    private final OpenSslAes256CbcCrypter crypter;

    private final TranscodeOverrideConfig overrideConfig;

    private final DynamicPropertyManager dynamicPropertyManager;

    @Value("${streaming.balancer-uri-mapping}")
    private String balancerUriMappingStr;

    @Value("${streaming.link-lifetime}")
    private Duration linkLifetime;

    @Value("${streaming.default-arg-filler.current}")
    private String currentDefaultArgFiller;

    @Value("${streaming.default-arg-filler.next}")
    private String nextDefaultArgFiller;

    @Value("${ffmpeg.version.current}")
    private String currentFFmpegVersion;

    @Value("${ffmpeg.version.next}")
    private String nextFFmpegVersion;

    @Value("${db.last-access-update-interval}")
    private Duration lastAccessUpdateInterval;

    @Value("${external-jetty.http.port}")
    private int extJettyPort;

    @Bean
    public DiskStreamingVideoInfoCache videoInfoCache(@Value("${streaming.memory-cache.ttl}") long ttl) {
        return new DiskStreamingVideoInfoCacheWithMemoryCache(daoConfig.videoInfoCacheDao(), lastAccessTimeUpdater(), ttl);
    }

    @Bean
    public MdsSegmentCache hlsSegmentCache(@Value("${streaming.memory-cache.ttl}") long ttl) {
        return new MdsSegmentCacheWithMemoryCache(daoConfig.segmentCacheDao(), mdsConfig.mds(), videoStreamingInternalClient(),
                lastAccessTimeUpdater(), ttl);
    }

    @Bean
    public VideoStreamingInternalClient videoStreamingInternalClient() {
        return new VideoStreamingInternalClient(
                internalHttpClientConfigurator().configure(),
                diskStreamingUrlBuilder(),
                extJettyPort
        );
    }

    @Bean
    @OverridableValuePrefix("vs-int")
    protected HttpClientConfigurator internalHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    protected IntervalRunner lastAccessTimeUpdater() {
        return new IntervalRunner(lastAccessUpdateInterval);
    }

    @Bean
    public DiskStreamingUrlBuilder diskStreamingUrlBuilder() {
        return new DiskStreamingUrlBuilder(crypter, HostMappingUtil.parse(balancerUriMappingStr), linkLifetime);
    }

    @Bean
    public DiskStreamingSegmentPrefetch diskStreamingSegmentPrefetch() {
        return new DiskStreamingSegmentPrefetch(diskStreamingUrlBuilder(), prefetchHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("prefetch")
    public HttpClientConfigurator prefetchHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    protected ForceFFmpeg3ForFormats forceFFmpeg3ForFormats() {
        return new ForceFFmpeg3ForFormats(dynamicPropertyManager);
    }

    @Bean
    public SourceMetaTransformer<MpfsSourceMeta> mpfsSourceMetaTransformer() {
        dynamicPropertyManager.addStaticFields(MpfsSourceMetaTransformer.class);
        return new MpfsSourceMetaTransformer(
                overrideConfig.transcodeOverridesRegistry(),
                forceFFmpeg3ForFormats(),
                new TransitionalValue<>(currentDefaultArgFiller, nextDefaultArgFiller),
                new TransitionalValue<>(currentFFmpegVersion, nextFFmpegVersion)
        );
    }

    static class DaoConfig {
        @Bean
        @OverridableValuePrefix("videostreaming")
        public DataSourceProperties dataSourceProperties() {
            return new DataSourceProperties();
        }
    }

    static class SegmentCacheIdProviderConfig {
        private final TranscodeOverrideConfig overrideConfig;

        private final HlsParams hlsParams;

        SegmentCacheIdProviderConfig (TranscodeOverrideConfig overrideConfig, HlsParams hlsParams) {
            this.overrideConfig = overrideConfig;
            this.hlsParams = hlsParams;
        }

        @Bean
        public SegmentCacheIdProvider cacheIdProvider() {
            return new SegmentCacheIdProvider(
                    overrideConfig.transcodeOverridesRegistry(),
                    hlsParams.getFFmpegParams(),
                    hlsParams.getFFmpegCommandParams()
            );
        }
    }
}
