package ru.yandex.chemodan.app.videostreaming.overrides;

import java.util.concurrent.atomic.AtomicBoolean;

import org.joda.time.Duration;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.TranscodeParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandParams;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.StreamingResourceParams;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.VideoInfoParams;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.zk.FakeInMemoryZkWrapper;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TranscodeParamOverriderProviderTest {
    private static final FFmpegParams FFMPEG_PARAMS = new FFmpegParams(
            Duration.standardSeconds(30), Duration.standardSeconds(5), Duration.standardSeconds(1), 9)
            .toBuilder()
            .outputLoggingEnabled(false)
            .build();

    private static final HlsFFmpegCommandParams FFMPEG_COMMAND_PARAMS =
            new HlsFFmpegCommandParams(2, 2, false, DataSize.MEGABYTE);

    private static final StreamingResourceParams STREAMING_RESOURCE_PARAMS =
            new StreamingResourceParams(false);

    private static final PassportUid FILTER_OWNER_UID = new PassportUid(1);

    private static final PassportUidOrZero MATCHING_OWNER_UID = new PassportUidOrZero(1);

    private static final PassportUidOrZero NON_MATCHING_OWNER_UID = new PassportUidOrZero(2);

    @Test
    public void overrideWhenFilterMatches() {
        assertParamsOverrides(
                MATCHING_OWNER_UID,
                STREAMING_RESOURCE_PARAMS.toBuilder()
                        .enableFullHd(true)
                        .build(),
                FFMPEG_PARAMS.toBuilder()
                        .outputLoggingEnabled(true)
                        .build(),
                FFMPEG_COMMAND_PARAMS.toBuilder()
                        .argFillerName("test-arg-filler")
                        .build()
        );
    }

    @Test
    public void doNotOverrideWhenFilterDoesNotMatch() {
        assertParamsOverrides(
                NON_MATCHING_OWNER_UID,
                STREAMING_RESOURCE_PARAMS,
                FFMPEG_PARAMS,
                FFMPEG_COMMAND_PARAMS
        );
    }

    private static void assertParamsOverrides(PassportUidOrZero actualOwnerUid,
            StreamingResourceParams expectedStreamingResourceParams,
            FFmpegParams expectedFFmpegParams,
            HlsFFmpegCommandParams expectedFFmpegCommandParams)
    {
        HlsParams hlsParams = consHlsParams();
        AtomicBoolean checked = new AtomicBoolean(false);
        hlsParams.consOverridable()
                .override(consOverrider(actualOwnerUid))
                .run(() -> {
                    Assert.equals(expectedStreamingResourceParams, hlsParams.getStreamingResourceParams());
                    Assert.equals(expectedFFmpegParams, hlsParams.getFFmpegParams());
                    Assert.equals(expectedFFmpegCommandParams, hlsParams.getFFmpegCommandParams());
                    checked.set(true);
                });
        Assert.isTrue(checked.get(), "Assertion was not executed");
    }

    private static HlsParams.Overrider consOverrider(PassportUidOrZero actualOwnerUid) {
        TranscodeOverridesRegistry registry = new TranscodeOverridesRegistry(new ZkPath("/"));
        new FakeInMemoryZkWrapper()
                .initialize(registry);
        registry.put(new TranscodeOverride("test-override", new RequestFilter(Cf.list(FILTER_OWNER_UID)),
                new TranscodeParams("test-arg-filler", true, true,
                        true, false, false, Option.empty()))
        );
        TranscodeParamOverriderProvider overriderProvider = new TranscodeParamOverriderProvider(registry);
        MpfsSourceMeta sourceMeta = new MpfsSourceMeta(MulcaId.valueOf("123", ""),
                Option.of(actualOwnerUid), Option.empty(), false, Option.empty());
        return overriderProvider.get(sourceMeta);
    }

    private static HlsParams consHlsParams() {
        return new HlsParams(
                new StreamingParams(MediaTime.seconds(5), 1, 3,
                        Duration.standardSeconds(5)),
                new VideoInfoParams(false),
                FFMPEG_PARAMS,
                FFMPEG_COMMAND_PARAMS,
                STREAMING_RESOURCE_PARAMS
        );
    }
}
