package ru.yandex.chemodan.videostreaming.framework.accesscheck;

import java.util.function.BiConsumer;
import java.util.function.BiFunction;

import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsRequest;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithIp;
import ru.yandex.chemodan.videostreaming.framework.web.IpProximityUtil;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Value
public class AccessParams {
    private static final Pattern2 INTERNAL_FILENAME_PATTERN = Pattern2.compile("(prefetch|local)\\d+\\..+");

    Option<IpAddress> expectedIpO;

    Option<IpAddress> requestIpO;

    Object sourceMeta;

    String filename;

    public static AccessParams fromRequests(HttpServletRequestX request, HlsRequest hlsRequest) {
        Object srcMeta = hlsRequest.getSourceMeta();
        Option<IpAddress> expectedIpO = srcMeta instanceof SourceMetaWithIp
                ? ((SourceMetaWithIp) srcMeta).getUserIp()
                : Option.empty();
        return new AccessParams(expectedIpO, request.getXRealIp(), srcMeta, hlsRequest.filename);
    }

    public boolean ipsAreEqual() {
        return applyOnBothPresent(Object::equals)
                .getOrElse(false);
    }

    public boolean ipsAreUnequal() {
        return applyOnBothPresent((expectedIp, actualIp) -> !expectedIp.equals(actualIp))
                .getOrElse(false);
    }

    public boolean ipsAreNearby(long maxIpv4Distance, long maxIpv6Distance) {
        return applyOnBothPresent((ip1, ip2) -> IpProximityUtil.areNearby(ip1, ip2, maxIpv4Distance, maxIpv6Distance))
                .getOrElse(false);
    }

    public <T> Option<T> applyOnBothPresent(BiFunction<IpAddress, IpAddress, T> biFunction) {
        return expectedIpO.filterMap(expectedIp ->
                requestIpO.map(requestIp -> biFunction.apply(expectedIp, requestIp))
        );
    }

    public void onBothIpPresent(BiConsumer<IpAddress, IpAddress> consumer) {
        expectedIpO.ifPresent(expectedIp ->
                requestIpO.ifPresent(requestIp -> consumer.accept(expectedIp, requestIp))
        );
    }

    public boolean expectedIpIsDefined() {
        return expectedIpO.isPresent();
    }

    public boolean isInternal() {
        return INTERNAL_FILENAME_PATTERN.matches(filename);
    }
}
