package ru.yandex.chemodan.videostreaming.framework.accesscheck;

import java.util.function.Predicate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.videostreaming.framework.web.IpMatcher;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ip.IpAddress;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class BasicAccessChecker implements AccessChecker {
    private static final DynamicProperty<Long> maxIpv4Distance =
            DynamicProperty.cons("streaming-access-check-nearby-distance-ipv4", 4096L);

    private static final DynamicProperty<Long> maxIpv6Distance =
            DynamicProperty.cons("streaming-access-check-nearby-distance-ipv6", 65536L);

    private final String name;

    private final AccessAction action;

    private final Predicate<AccessParams> predicate;

    private BasicAccessChecker(String name, AccessAction action, Predicate<AccessParams> predicate) {
        this.name = name;
        this.action = action;
        this.predicate = predicate;
    }

    public static BasicAccessChecker allow(String name, Predicate<AccessParams> predicate) {
        return new BasicAccessChecker(name, AccessAction.ALLOW, predicate);
    }

    public static BasicAccessChecker deny(String name, Predicate<AccessParams> predicate) {
        return new BasicAccessChecker(name, AccessAction.DENY, predicate);
    }

    public static BasicAccessChecker allowEqualIps() {
        return allow("allow-equal", AccessParams::ipsAreEqual);
    }

    public static BasicAccessChecker denyUnequalIps() {
        return deny("deny-unequal", AccessParams::ipsAreUnequal);
    }

    public static BasicAccessChecker allowInternalFilenames() {
        return allow("allow-internal-filename", AccessParams::isInternal);
    }

    public static BasicAccessChecker allowIfAnyIpMatches(String name, IpMatcher ipMatcher) {
        return ifAnyIpMatches(name, AccessAction.ALLOW, ipMatcher);
    }

    public static BasicAccessChecker denyIfAnyIpMatches(String name, IpMatcher ipMatcher) {
        return ifAnyIpMatches(name, AccessAction.DENY, ipMatcher);
    }

    private static BasicAccessChecker ifAnyIpMatches(String name, AccessAction action, IpMatcher ipMatcher) {
        Predicate<Option<IpAddress>> ipMatchesP = ipO -> ipO.isMatch(ipMatcher::matches);
        return new BasicAccessChecker(
                name,
                action,
                params -> ipMatchesP.test(params.getExpectedIpO()) || ipMatchesP.test(params.getRequestIpO())
        );
    }

    public static BasicAccessChecker alwaysAllow() {
        return constant(AccessAction.ALLOW);
    }

    public static BasicAccessChecker alwaysDeny() {
        return constant(AccessAction.DENY);
    }

    public static BasicAccessChecker alwaysNone() {
        return constant(AccessAction.NONE);
    }

    static BasicAccessChecker constant(AccessAction action) {
        return new BasicAccessChecker(
                String.format("%s-always", action.name().toLowerCase()),
                action,
                Function1B.trueF()
        );
    }

    public static BasicAccessChecker allowNearbyIps() {
        return allow("allow-nearby-ips",
                params -> params.ipsAreNearby(maxIpv4Distance.get(), maxIpv6Distance.get())
        );
    }

    public String getName() {
        return name;
    }

    @Override
    public Result getResult(AccessParams params) {
        return new Result(name, getAction(params), params);
    }

    AccessAction getAction(AccessParams params) {
        return predicate.test(params) ? action : AccessAction.NONE;
    }
}
