package ru.yandex.chemodan.videostreaming.framework.cachingproxy;

import java.io.IOException;
import java.io.OutputStream;

import org.apache.commons.io.output.CountingOutputStream;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class RetryingUpstream<T> implements Upstream<T> {
    private static final Logger logger = LoggerFactory.getLogger(RetryingUpstream.class);

    private final Upstream<T> upstream;

    private final int retryCount;

    public RetryingUpstream(Upstream<T> upstream, int retryCount) {
        Validate.ge(retryCount, 0);
        this.upstream = upstream;
        this.retryCount = retryCount;
    }

    @Override
    public long fetchContentLength(T resourceId) throws ResourceNotFoundException {
        for(int n = 0; n <= retryCount; n++) {
            try {
                return upstream.fetchContentLength(resourceId);
            } catch (ResourceNotFoundException e) {
                throw e;
            } catch (RuntimeException e) {
                if (n >= retryCount) {
                    throw e;
                }
                logger.warn("Retrying on error from upstream", e);
            }
        }

        throw new IllegalStateException("MUST be unreachable");
    }

    @Override
    public void downloadTo(T resourceId, final ByteRange baseRange, OutputStream to) throws ResourceNotFoundException {
        CountingOutputStream out = new CountingOutputStream(to);
        for(int n = 0; n <= retryCount; n++) {
            ByteRange range = baseRange.shiftStart(out.getByteCount());
            try {
                upstream.downloadTo(resourceId, range, out);
                return;
            } catch (ResourceNotFoundException e) {
                throw e;
            } catch (IOException | RuntimeException e) {
                if (n >= retryCount) {
                    throw ExceptionUtils.translate(e);
                }

                logger.warn("Retrying on error from upstream, bytes received so far = {}",
                        out.getByteCount(), e);
            }
        }
    }

    @Override
    public T parse(String value) {
        return upstream.parse(value);
    }

    @Override
    public String serialize(T value) {
        return upstream.serialize(value);
    }
}
