package ru.yandex.chemodan.videostreaming.framework.cachingproxy;

import java.io.IOException;
import java.io.OutputStream;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.servlet.http.HttpServletRequest;

import org.eclipse.jetty.io.EofException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.util.AsyncCacheableInputStreamSource;
import ru.yandex.chemodan.videostreaming.framework.util.TimeAndSuccessLogUtil;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.CacheUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * @author lemeh
 */
public class StorageProxyManager<T> {
    private final Upstream<T> upstream;
    private final Cache<T, Long> contentLengthCache;
    private final Cache<ChunkId, Chunk> dataCache;
    private final ExecutorService executor;

    private final long chunkSize;

    public StorageProxyManager(Upstream<T> upstream, int cacheSize, DataSize chunkSize) {
        this.upstream = upstream;
        this.contentLengthCache = CacheUtils.newLru(cacheSize);
        this.dataCache = CacheUtils.newLru(cacheSize);
        this.chunkSize = chunkSize.toBytes();
        this.executor = Executors.newFixedThreadPool(100);
    }

    Session getSession(HttpServletRequest req) {
        String value = StringUtils.removeEnd(
                StringUtils.removeStart(req.getPathInfo(), "/"),
                "/"
        );
        return getSession(upstream.parse(value));
    }

    Session getSession(T resourceId) {
        return new Session(resourceId);
    }

    public class Session {
        final Logger logger = LoggerFactory.getLogger(Session.class);

        final long chunkSize = StorageProxyManager.this.chunkSize;

        final T resourceId;

        final long contentLength;

        Session(T resourceId) {
            this.resourceId = resourceId;
            this.contentLength = contentLengthCache.getFromCacheSome(resourceId, upstream::fetchContentLength);
        }

        void transferTo(OutputStream os, DownloadRange range) {
            logger.debug("Stream request: resourceId={}, start={}, end={}",
                    resourceId, range.start(), range.endO);
            try {
                transferToUnsafe(os, range);
            } catch (RuntimeEofException e) {
                logger.warn("Stream failed due to unexpected EOF: {}", getAllMessages(e));
            } catch (Exception e) {
                logger.warn("Stream failed: {}", getAllMessages(e), e);
                throw translateException(e);
            }
        }

        private void transferToUnsafe(OutputStream os, DownloadRange range) {
            ByteRange byteRange = range.toByteRange(contentLength);
            int minIndex = (int) (byteRange.startInclusive / chunkSize);
            int maxIndex = Double.valueOf(Math.ceil(byteRange.endExclusive / (double) chunkSize)).intValue();
            Cf.range(minIndex, maxIndex)
                    .iterator()
                    .map(this::consByteRange)
                    .map(this::consChunkId)
                    .map(this::getChunk)
                    .forEachRemaining(chunk -> chunk.transferTo(os, byteRange));
        }

        private ByteRange consByteRange(int index) {
            return new ByteRange(chunkSize * index, Math.min(chunkSize * (index + 1), contentLength));
        }

        private ChunkId consChunkId(ByteRange range) {
            return new ChunkId(resourceId, range);
        }

        private Chunk getChunk(ChunkId id) {
            return dataCache.getFromCacheSome(id, this::consChunkReceivingFromUpstream);
        }

        private Chunk consChunkReceivingFromUpstream(ChunkId id) {
            Chunk chunk = new Chunk(id);
            executor.submit(RequestIdStack.withCurrentRequestIdF(
                    () -> {
                        try {
                            chunk.receiveFromUpstream();
                        } catch (RuntimeException ex) {
                            dataCache.removeFromCache(id);
                        }
                    }
            ));
            return chunk;
        }
    }

    private final class ChunkId extends DefaultObject {
        final T resourceId;

        final ByteRange range;

        ChunkId(T resourceId, ByteRange range) {
            this.resourceId = resourceId;
            this.range = range;
        }
    }

    private class Chunk extends DefaultObject {
        final Logger logger = LoggerFactory.getLogger(Session.class);

        final ChunkId id;

        final AsyncCacheableInputStreamSource iss;

        Chunk(ChunkId id) {
            this.id = id;
            this.iss = new AsyncCacheableInputStreamSource(id.range.lengthInt());
        }

        void receiveFromUpstream() {
            iss.receiveFrom(
                    out -> TimeAndSuccessLogUtil.runLoggingTimeAndSuccess(
                            () -> doReceiveFromUpstream(out),
                            "Receive chunk from storage with size = " + id.range.length(),
                            logger,
                            "chunk_size", id.range.length()
                    )
            );
        }

        private void doReceiveFromUpstream(OutputStream out) {
            try {
                upstream.downloadTo(id.resourceId, id.range, out);
            } catch (IOException | RuntimeException e) {
                // TODO: add retries
                throw translateException(e);
            }
        }

        void transferTo(OutputStream os, ByteRange downloadRange) {
            logger.debug("Streaming {}", this);
            try (InputStreamX in = openInputStream(downloadRange)) {
                IoUtils.copy(in, os);
            } catch (IOException | RuntimeIoException e) {
                throw translateException(e);
            }
        }

        private InputStreamX openInputStream(ByteRange downloadRange) {
            ByteRange localRange = id.range.intersectWith(downloadRange).minus(id.range.startInclusive);
            return iss.getInputStreamX((int) localRange.startInclusive, localRange.lengthInt());
        }

        @Override
        public String toString() {
            return id.range.toString();
        }
    }

    private static RuntimeException translateException(Exception e) {
        Option<Throwable> causeO = Option.ofNullable(e.getCause());
        if (e instanceof EofException) {
            return new RuntimeEofException(e);
        } else if (causeO.isMatch(t -> t instanceof EofException)) {
            return new RuntimeEofException(causeO.get());
        } else {
            return ru.yandex.misc.ExceptionUtils.translate(e);
        }
    }

    private static Object getAllMessages(Exception e) {
        return ru.yandex.misc.ExceptionUtils.getAllMessages(e);
    }

    private static class RuntimeEofException extends RuntimeIoException {
        public RuntimeEofException(Throwable cause) {
            super(cause);
        }
    }
}
