package ru.yandex.chemodan.videostreaming.framework.cachingproxy;

import java.io.IOException;
import java.util.function.BiConsumer;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * @author lemeh
 */
public class StorageProxyServlet<T> extends HttpServlet {
    private static final SetF<String> ALLOWED_METHODS = Cf.set("GET", "HEAD");

    private final StorageProxyManager<T> storageProxyManager;

    private final BiConsumer<HttpServletRequestX, Throwable> onErrorCallback;

    public StorageProxyServlet(StorageProxyManager<T> storageProxyManager,
            BiConsumer<HttpServletRequestX, Throwable> onErrorCallback)
    {
        this.storageProxyManager = storageProxyManager;
        this.onErrorCallback = onErrorCallback;
    }

    @Override
    protected final void service(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        service(HttpServletRequestX.wrap(req), HttpServletResponseX.wrap(resp));
    }

    private void service(HttpServletRequestX req, HttpServletResponseX resp) throws IOException {
        try {
            serviceUnsafe(req, resp);
        } catch (Throwable ex) {
            onErrorCallback.accept(req, ex);
            throw ex;
        }
    }

    private void serviceUnsafe(HttpServletRequestX req, HttpServletResponseX resp) throws IOException {
        if (!ALLOWED_METHODS.containsTs(req.getMethod())) {
            Validate.fail();
        }

        StorageProxyManager.Session session;
        try {
            session = getSession(req);
            Option<DownloadRange> rangeO = Option.when("GET".equals(req.getMethod()), extractRangeOrFull(req));
            addCommonHeadersAndSetStatus(resp, session, rangeO);
            if ("GET".equals(req.getMethod())) {
                session.transferTo(resp.getOutputStream(), rangeO.get());
            }
        } catch(ResourceNotFoundException ex) {
            resp.setStatus(HttpStatus.SC_404_NOT_FOUND);
        }
    }

    private static DownloadRange extractRangeOrFull(HttpServletRequest req) {
        return new HttpServletRequestX(req)
                    .getHeaderO("Range")
                    .map(DownloadRange::parse)
                    .getOrElse(DownloadRange.full());
    }

    private static void addCommonHeadersAndSetStatus(HttpServletResponse resp, StorageProxyManager.Session session,
            Option<DownloadRange> rangeO)
    {
        resp.addHeader("Content-Type", "application/octet-stream");
        resp.addHeader("Accept-Ranges", "bytes");

        if (rangeO.isPresent() && rangeO.get().isPartial(session.contentLength)) {
            resp.addHeader("Content-Length", "" + rangeO.get().length(session.contentLength));
            resp.addHeader("Content-Range", "bytes " + rangeO.get().toContentRange(session.contentLength));
            resp.setStatus(HttpStatus.SC_206_PARTIAL_CONTENT);
        } else {
            resp.addHeader("Content-Length", "" + session.contentLength);
            resp.setStatus(HttpStatus.SC_200_OK);
        }
    }

    private StorageProxyManager<T>.Session getSession(HttpServletRequest req) {
        return storageProxyManager.getSession(req);
    }
}
