package ru.yandex.chemodan.videostreaming.framework.config;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.BasicAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.CompositeAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.util.HbfApiClient;
import ru.yandex.chemodan.videostreaming.framework.web.BannedIpRegistry;
import ru.yandex.chemodan.videostreaming.framework.web.IpByNetworkMatcher;
import ru.yandex.commune.dynproperties.DynamicPropertiesContextConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({BannedIpConfig.class, DynamicPropertiesContextConfiguration.class})
public class AccessCheckConfig {
    @Value("${streaming.ip-access-check.yandex-subnets-macro}")
    private String yandexSubnetsMacro;

    @Value("${hbf.host}")
    private String hbfHost;

    @Autowired
    private BannedIpRegistry bannedIpRegistry;

    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    @Bean
    public CompositeAccessChecker compositeAccessChecker(@Qualifier("allowYandexIpAccessChecker") BasicAccessChecker allowYandexIpAccessChecker) {
        CompositeAccessChecker compositeChecker = new CompositeAccessChecker(
                allowYandexIpAccessChecker,
                BasicAccessChecker.allowEqualIps(),
                BasicAccessChecker.allowInternalFilenames(),
                BasicAccessChecker.allowNearbyIps(),
                BasicAccessChecker.denyUnequalIps(),
                bannedIpAccessChecker()
        );
        compositeChecker.register(dynamicPropertyManager);
        return compositeChecker;
    }

    @Bean
    protected BasicAccessChecker allowYandexIpAccessChecker(IpByNetworkMatcher yandexIpMatcher) {
        return BasicAccessChecker.allowIfAnyIpMatches("allow-yandex", yandexIpMatcher);
    }

    @Bean
    protected BasicAccessChecker bannedIpAccessChecker() {
        return BasicAccessChecker.denyIfAnyIpMatches("deny-banned", ip -> bannedIpRegistry.has(ip));
    }

    @Bean
    public IpByNetworkMatcher yandexIpMatcher(HbfApiClient hbfApiClient) {
        return new IpByNetworkMatcher(() -> hbfApiClient.expandMacro(yandexSubnetsMacro));
    }

    @Bean
    public HbfApiClient hbfApiClient() {
        return new HbfApiClient(hbfHost, hbfHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("hbf")
    public HttpClientConfigurator hbfHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @PostConstruct
    protected void addDynamicProperties() {
        dynamicPropertyManager.addStaticFields(BasicAccessChecker.class);
    }
}
