package ru.yandex.chemodan.videostreaming.framework.config;

import java.net.URI;
import java.util.UUID;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportSelector;
import org.springframework.core.type.AnnotationMetadata;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.http.YandexCloudRequestIdHandler;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.RetryingUpstream;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.StorageProxyManager;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.StorageProxyServlet;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.Upstream;
import ru.yandex.chemodan.videostreaming.framework.cachingproxy.UpstreamIdProvider;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegSourceExceptionRegistry;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsFFmpegSourceProvider;
import ru.yandex.chemodan.videostreaming.framework.util.RequestIdHelper;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
public class HlsCachingSourceProxyConfig<SourceMetaT, UpstreamIdT> {
    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private HlsCachingSourceProxyParams params;

    @Autowired
    @SuppressWarnings({"SpringJavaAutowiringInspection", "SpringAutowiredFieldsWarningInspection"})
    private Upstream<UpstreamIdT> upstream;

    @Autowired
    @SuppressWarnings({"SpringJavaAutowiringInspection", "SpringAutowiredFieldsWarningInspection"})
    private UpstreamIdProvider<SourceMetaT, UpstreamIdT> upstreamIdProvider;

    @Bean
    public HlsFFmpegSourceProvider<SourceMetaT> hlsFFmpegSourceProvider() {
        return (sourceMeta, sessionId) -> URI.create(
                String.format("http://localhost:%d/%s/%s?session=%s&%s=%s",
                        params.proxyPort,
                        params.proxyUrlPrefix,
                        upstream.serialize(upstreamIdProvider.getUpstreamId(sourceMeta)),
                        sessionId,
                        YandexCloudRequestIdHandler.PARENT_RID_PARAM_NAME,
                        RequestIdHelper.getCurrentRequestId()
                )
        );
    }

    @Bean
    public StorageProxyServlet<UpstreamIdT> storageProxyServlet() {
        RetryingUpstream<UpstreamIdT> retryingUpstream = new RetryingUpstream<>(upstream, params.upstreamRetryCount);
        return new StorageProxyServlet<>(
                new StorageProxyManager<>(retryingUpstream, params.cacheSize, params.cacheDataSize),
                (req, ex) -> {
                    Option<UUID> sessionIdO = req.getParameterO("session").map(UUID::fromString);
                    if (!sessionIdO.isPresent()) {
                        return;
                    }

                    FFmpegSourceExceptionRegistry.INSTANCE.setException(sessionIdO.get(), ex);
                }
        );
    }

    public static class ConfigImportSelector implements ImportSelector, BeanFactoryAware {
        private BeanFactory beanFactory;

        @Override
        public String[] selectImports(AnnotationMetadata importingClassMetadata) {
            try {
                beanFactory.getBean(Upstream.class);
                return new String[]{HlsCachingSourceProxyConfig.class.getName()};
            } catch (NoSuchBeanDefinitionException ex) {
                return new String[0];
            }
        }

        @Override
        public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
            this.beanFactory = beanFactory;
        }
    }
}
