package ru.yandex.chemodan.videostreaming.framework.config;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;
import javax.cache.expiry.ModifiedExpiryPolicy;

import org.apache.ignite.cache.affinity.rendezvous.RendezvousAffinityFunction;
import org.apache.ignite.cache.eviction.lru.LruEvictionPolicyFactory;
import org.apache.ignite.configuration.CacheConfiguration;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.config.IgniteConfig;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.CompositeAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpeg;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegExecutor;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.TranscodingInventory;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsFFmpegSourceProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsManager;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParamsBySourceOverriderProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsUrlBuilder;
import ru.yandex.chemodan.videostreaming.framework.hls.MediaSegmentation;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.HlsFFmpegCommandProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.HlsStreamingManager;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.argfiller.FFmpegArgFillerRegistry;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.argfiller.FFmpegArgFillerZkRegistry;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.HlsSegmentCache;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.HlsSegmentPrefetch;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.HlsSegmentPrefetchTrigger;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaConverter;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaTransformer;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.HlsFFmpegVideoInfoProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.HlsVideoInfoManager;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.HlsVideoInfoProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.StreamingResourceProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.WithStatsVideoInfoProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.caching.HlsCachingVideoInfoProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.caching.VideoInfoCache;
import ru.yandex.chemodan.videostreaming.framework.stidblocker.StidBlocker;
import ru.yandex.chemodan.videostreaming.framework.stidblocker.StidBlockerConfiguration;
import ru.yandex.chemodan.videostreaming.framework.web.BannedIpUpdater;
import ru.yandex.chemodan.videostreaming.framework.web.HlsRequestParser;
import ru.yandex.chemodan.videostreaming.framework.web.HlsServlet;
import ru.yandex.chemodan.videostreaming.framework.web.HlsVideoInfoServlet;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.misc.dataSize.DataSize;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import({HlsCachingSourceProxyConfig.ConfigImportSelector.class, AccessCheckConfig.class, StidBlockerConfiguration.class})
public class HlsConfig<T> {
    private final HlsOutputConfig outputConfig;

    private final IgniteConfig igniteConfig;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private HlsParams hlsParams;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private FFmpegVersions ffmpegVersions;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection"})
    private Optional<HlsSegmentPrefetch<T>> segmentPrefetch;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection"})
    private Optional<HlsSegmentCache<T>> segmentCache;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection"})
    private Optional<VideoInfoCache<T>> videoInfoCache;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection"})
    private Optional<SourceMetaConverter<T>> sourceMetaConverter;

    @Autowired
    @VideoInfoQualifier
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private SourceMetaParser<T> videoInfoSourceMetaParser;

    @Autowired
    @StreamingQualifier
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private SourceMetaParser<T> sourceMetaParser;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection", "SpringJavaAutowiringInspection"})
    private HlsFFmpegSourceProvider<T> ffmpegSourceProvider;

    @Autowired
    @SuppressWarnings({"SpringJavaAutowiringInspection", "SpringAutowiredFieldsWarningInspection"})
    private HlsSegmentPrefetchTrigger segmentPrefetchTrigger;

    @Autowired
    @SuppressWarnings({"SpringJavaAutowiringInspection", "SpringAutowiredFieldsWarningInspection"})
    private HlsUrlBuilder<T> urlBuilder;

    @Autowired
    private Optional<HlsParamsBySourceOverriderProvider<T>> hlsParamsOverriderProvider;

    @Autowired
    private Optional<SourceMetaTransformer<T>> sourceMetaTransformer;

    @Autowired
    @SuppressWarnings({"SpringJavaAutowiringInspection", "SpringAutowiredFieldsWarningInspection"})
    private DynamicPropertyManager dynamicPropertyManager;

    @Autowired
    @SuppressWarnings({"SpringAutowiredFieldsWarningInspection"})
    private Optional<BannedIpUpdater> bannedIpUpdater;

    @Autowired
    @SuppressWarnings("SpringJavaAutowiringInspection")
    protected HlsConfig(HlsOutputConfig outputConfig, IgniteConfig igniteConfig) {
        this.outputConfig = outputConfig;
        this.igniteConfig = igniteConfig;
    }

    @Value("${hls.cors-whitelist}")
    private String corsWhitelist;

    @Value("${streaming.prefetch-cache-size}")
    private int prefetchCacheSize;

    @Value("${streaming.max-ip-distance}")
    private DataSize maxIpDistance;

    @Value("${streaming.transcodings.cache.max-size}")
    private int transcodingsCacheMaxSize;

    @Value("${streaming.transcodings.cache.ttl}")
    private Duration transcodingsCacheTtl;

    @Autowired
    private CompositeAccessChecker compositeAccessChecker;

    @Autowired
    private StidBlocker stidBlocker;

    @Bean
    protected CacheConfiguration<String, TranscodingInventory.Transcodings> transcodingsCacheConfig() {
        CacheConfiguration<String, TranscodingInventory.Transcodings> cacheConfig =
                new CacheConfiguration<>("transcodings");
        cacheConfig.setAffinity(new RendezvousAffinityFunction());
        cacheConfig.setOnheapCacheEnabled(true);
        cacheConfig.setEvictionPolicyFactory(new LruEvictionPolicyFactory<>(transcodingsCacheMaxSize));
        cacheConfig.setExpiryPolicyFactory(ModifiedExpiryPolicy.factoryOf(getTranscodingsCacheTtl()));
        return cacheConfig;
    }

    private javax.cache.expiry.Duration getTranscodingsCacheTtl() {
        return new javax.cache.expiry.Duration(TimeUnit.MILLISECONDS, transcodingsCacheTtl.getMillis());
    }

    @Bean
    public TranscodingInventory transcodingInventory() {
        dynamicPropertyManager.addStaticFields(TranscodingInventory.class);
        return new TranscodingInventory(
                igniteConfig.igniteSpringBean(),
                transcodingsCacheConfig(),
                hlsParams.getFFmpegParams()
        );
    }

    @Bean
    public FFmpegExecutor ffmpegExecutor() {
        return new FFmpegExecutor(transcodingInventory());
    }

    @Bean
    public FFmpeg ffmpeg() {
        dynamicPropertyManager.addStaticFields(FFmpegExecutor.class);
        return new FFmpeg(ffmpegVersions, hlsParams.getFFmpegParams(), ffmpegExecutor());
    }

    @Bean
    public HlsFFmpegCommandProvider<T> ffmpegCommandProvider() {
        return new HlsFFmpegCommandProvider<>(
                ffmpegArgFillerRegistry(),
                ffmpegSourceProvider,
                outputConfig.hlsOutputProvider(),
                hlsParams.getFFmpegCommandParams()
        );
    }

    @Bean
    public FFmpegArgFillerRegistry ffmpegArgFillerRegistry() {
        return new FFmpegArgFillerRegistry();
    }

    @Bean
    public FFmpegArgFillerZkRegistry ffmpegArgFillerZkRegistry(
            @Qualifier("zkRoot") ZkPath zkRoot,
            ZkManager zkManager)
    {
        FFmpegArgFillerZkRegistry zkRegistry = new FFmpegArgFillerZkRegistry(
                zkRoot.child("videostreaming/ffmpeg-arg-fillers"),
                ffmpegArgFillerRegistry()
        );
        zkManager.addClient(zkRegistry);
        return zkRegistry;
    }

    @Bean
    public HlsVideoInfoProvider<T> videoInfoProvider() {
        return Option.x(videoInfoCache)
                .map(this::consCachingVideoInfoProvider)
                .getOrElse(this::ffmpegVideoInfoProvider);
    }

    private HlsVideoInfoProvider<T> consCachingVideoInfoProvider(VideoInfoCache<T> cache) {
        return new WithStatsVideoInfoProvider<>(
                new HlsCachingVideoInfoProvider<>(cache, ffmpegVideoInfoProvider(), hlsParams.getVideoInfoParams())
        );
    }

    @Bean
    protected HlsVideoInfoProvider<T> ffmpegVideoInfoProvider() {
        return new HlsFFmpegVideoInfoProvider<>(ffmpeg(), ffmpegSourceProvider);
    }

    @Bean
    @Lazy
    public HlsRequestParser<T> hlsRequestParser() {
        dynamicPropertyManager.addStaticFields(HlsRequestParser.class);
        return new HlsRequestParser<>(sourceMetaParser, compositeAccessChecker,
                bannedIpUpdater.orElse(BannedIpUpdater.noop()), stidBlocker);
    }

    @Bean
    @Lazy
    public HlsServlet<T> hlsServlet() {
        dynamicPropertyManager.addStaticFields(HlsServlet.class);
        return new HlsServlet<>(
                hlsRequestParser(),
                hlsManager(),
                sourceMetaConverter.orElseGet(SourceMetaConverter::defaultConverter),
                Cf.x(corsWhitelist.split(","))
                        .map(String::toLowerCase)
                        .unique(),
                hlsParams,
                getHlsParamsOverriderProvider()
        );
    }

    @SuppressWarnings("ConstantConditions")
    private HlsSegmentCache<T> segmentCache() {
        return segmentCache.get();
    }

    @Bean
    public HlsStreamingManager<T> streamingManager() {
        return new HlsStreamingManager<>(
                ffmpeg(),
                ffmpegCommandProvider(),
                segmentCache(),
                segmentPrefetch.orElse(HlsSegmentPrefetch.noop()),
                segmentPrefetchTrigger,
                hlsParams.getStreamingParams(),
                prefetchCacheSize
        );

    }

    @Bean
    protected HlsManager<T> hlsManager() {
        return new HlsManager<>(
                streamingManager(),
                streamingResourceProvider(),
                hlsParams,
                urlBuilder,
                segmentCache(),
                videoInfoCache.get()
        );
    }

    @Bean
    @Lazy
    public HlsVideoInfoServlet<T> hlsVideoInfoServlet() {
        return new HlsVideoInfoServlet<>(
                hlsVideoInfoManager(),
                videoInfoSourceMetaParser,
                getHlsParamsOverriderProvider(),
                hlsParams
        );
    }

    private HlsParamsBySourceOverriderProvider<T> getHlsParamsOverriderProvider() {
        return hlsParamsOverriderProvider.orElse(HlsParamsBySourceOverriderProvider.noop());
    }

    @Bean
    protected HlsVideoInfoManager<T> hlsVideoInfoManager() {
        return new HlsVideoInfoManager<>(streamingResourceProvider(), urlBuilder,
                sourceMetaTransformer.orElse(SourceMetaTransformer.noop())
        );
    }

    @Bean
    public StreamingResourceProvider<T> streamingResourceProvider() {
        return new StreamingResourceProvider<>(videoInfoProvider(), hlsParams.getStreamingResourceParams());
    }

    @PostConstruct
    public void initStaticDynamicProperties() {
        dynamicPropertyManager.addStaticFields(MediaSegmentation.class);
        dynamicPropertyManager.addStaticFields(HlsManager.class);
    }
}
