package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import java.io.IOException;
import java.io.InputStream;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import com.google.common.collect.EvictingQueue;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.WaitInCycle;
import ru.yandex.chemodan.videostreaming.framework.config.FFmpegVersions;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeInfo;
import ru.yandex.chemodan.videostreaming.framework.hls.stats.FFUtilStats;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.util.CommonThreadPoolHolder;
import ru.yandex.chemodan.videostreaming.framework.util.TimeAndSuccessLogUtil;
import ru.yandex.chemodan.videostreaming.framework.util.system.AppCpuTime;
import ru.yandex.chemodan.videostreaming.framework.web.HlsErrorSource;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ExecUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFmpeg {
    public static final String FFMPEG_LOGGER_NAME = "ffmpeg-output";

    private static final Logger logger = LoggerFactory.getLogger(FFmpeg.class);

    private static final Logger ffmpegLogger = LoggerFactory.getLogger(FFMPEG_LOGGER_NAME);

    private static final String FFPROBE_ENTRIES = Tuple2List.<String, ListF<String>>fromPairs(
            "format", Cf.list("bit_rate", "duration", "start_time", "format_name", "probe_score"),
            "stream", Cf.list("codec_type", "index", "codec_name", "bit_rate", "sample_rate", "channels",
                    "width", "height", "display_aspect_ratio", "r_frame_rate", "avg_frame_rate", "channel_layout",
                    "duration"),
            "stream_tags", Cf.list("rotate"),
            "stream_disposition", Cf.list(),
            "format_tags", Cf.list("creation_time"),
            "side_data", Cf.list()
    ).map2(strings -> strings.mkString(","))
            .map((key, entries) -> key + "=" + entries)
            .mkString(":");

    private static final Duration WAIT_FOR_SLEEP_DURATION = Duration.standardSeconds(1);

    private static final int MAX_STDERR_LINE_COUNT = 25;

    private static final int MAX_STDERR_LINE_SIZE = 2000;

    private final FFmpegExecutor executor;

    public final FFmpegVersions ffmpegVersions;

    public final FFmpegParams defaultParams;

    public FFmpeg(FFmpegVersions ffmpegVersions, FFmpegParams defaultParams, FFmpegExecutor executor) {
        this.ffmpegVersions = ffmpegVersions;
        this.defaultParams = defaultParams;
        this.executor = executor;
    }

    public FFPaths getPaths() {
        return ffmpegVersions.get(defaultParams.getFfmpegVersion());
    }

    public MediaInfo videoInformation(FFmpegSource source) {
        return TimeAndSuccessLogUtil.execLoggingTimeAndSuccess(() -> runFFprobe(source), "FFprobe command", logger);
    }

    private MediaInfo runFFprobe(FFmpegSource source) {
        ExecResult r = runFFprobeExecResultCompact(source);
        if (!r.isSuccess()) {
            throw convertStderrToException(source, r.getStderr());
        }
        return new MediaInfo(FFprobeInfo.parse(r.getStdout()), r.getStdout());
    }

    private ExecResult runFFprobeExecResultCompact(FFmpegSource source) {
        return runFFprobeExecResult(source, true);
    }

    public ExecResult runFFprobeExecResult(FFmpegSource source) {
        return runFFprobeExecResult(source, false);
    }

    private ExecResult runFFprobeExecResult(FFmpegSource source, boolean compact) {
        AppCpuTime.Snapshot startCpuTime = AppCpuTime.getCpuTime();
        try {
            ProcessBuilder cmd = buildFFprobeCommand(getPaths().getFFprobe().getAbsolutePath(), source, compact);
            logger.info("Executing {}", toString(cmd));
            return ExecUtils.executeGrabbingOutput(cmd);
        } finally {
            FFUtilStats.getCurrent().setFFprobeAppCpuTime(
                    new AppCpuTime.Interval(startCpuTime, AppCpuTime.getCpuTime())
            );
        }
    }

    static ProcessBuilder buildFFprobeCommand(String ffprobePath, FFmpegSource source, boolean compact) {
        return new ProcessBuilder(ffprobePath,
                "-v", "warning", "-hide_banner", "-print_format", "json=compact=" + (compact ? 1 : 0),
                "-show_format", "-show_streams", "-show_entries", FFPROBE_ENTRIES, source.getUriStr()
        );
    }

    @SuppressWarnings("UnusedReturnValue")
    public Session execute(FFmpegCommand command) {
        return execute(command, transcodingState -> { /* do nothing */ });
    }

    public Session execute(FFmpegCommand command, FFmpegStatsHandler statsHandler) {
        return execute(command, statsHandler, e -> { /* do nothing */ });
    }

    public Session execute(FFmpegCommand command, FFmpegStatsHandler statsHandler,
            Consumer<RuntimeException> errorHandler)
    {
        return execute(command, statsHandler, errorHandler, defaultParams);
    }

    public Session execute(FFmpegCommand command, FFmpegStatsHandler statsHandler,
            Consumer<RuntimeException> errorHandler, FFmpegParams params)
    {
        return new Execution(command, statsHandler, errorHandler, params)
                .executeAsync();
    }

    private static String toString(ProcessBuilder processBuilder) {
        return StringUtils.join(processBuilder.command(), " ");
    }

    public class Session extends DefaultObject {
        private volatile Option<AppCpuTime.Snapshot> startCpuTimeO = Option.empty();

        private final CompletableFuture<Process> processFuture = new CompletableFuture<>();

        private final CompletableFuture<Void> finishedFuture = new CompletableFuture<>();

        public void destroyForciblyQuietly() {
            if (!getProcess().isAlive()) {
                return;
            }

            try {
                getProcess().destroyForcibly();
            } catch (RuntimeException ex) {
                logger.warn("Error while stopping FFmpeg", ex);
            }
        }

        @SuppressWarnings("UnusedReturnValue")
        public Process waitForProcess() {
            return getProcess();
        }

        public Process getProcess() {
            return getFutureValue(processFuture);
        }

        private void setProcess(Process process) {
            startCpuTimeO = Option.of(AppCpuTime.getCpuTime());
            processFuture.complete(process);
        }

        private void setStartProcessException(Throwable exception) {
            processFuture.completeExceptionally(exception);
        }

        public boolean isFinished() {
            return finishedFuture.isDone();
        }

        public void setExecutionFinished() {
            finishedFuture.complete(null);
        }

        public void setExecutionException(Throwable ex) {
            finishedFuture.completeExceptionally(ex);
        }

        public Option<AppCpuTime.Interval> getAppCpuTimeO() {
            return startCpuTimeO.map(
                    startCpuTime -> new AppCpuTime.Interval(startCpuTime, AppCpuTime.getCpuTime())
            );
        }

        private <T> T getFutureValue(CompletableFuture<T> future) {
            try {
                return future.get();
            } catch (InterruptedException e) {
                throw ExceptionUtils.translate(e);
            } catch (ExecutionException e) {
                throw ExceptionUtils.translate((Exception) e.getCause());
            }
        }
    }

    class Execution extends DefaultObject {
        final Session session = new Session();

        final FFmpegCommand command;

        final FFmpegStatsHandler statsHandler;

        final Consumer<RuntimeException> errorHandler;

        final FFmpegParams params;

        final ProcessBuilder processBuilder;

        final Queue<String> stderrContents = EvictingQueue.create(MAX_STDERR_LINE_COUNT);

        volatile Option<FFmpegStats> lastStatsO = Option.empty();

        volatile Option<Instant> startTimeO = Option.empty();

        Execution(FFmpegCommand command, FFmpegStatsHandler statsHandler, Consumer<RuntimeException> errorHandler,
                FFmpegParams params)
        {
            this.command = command;
            this.statsHandler = statsHandler;
            this.errorHandler = errorHandler;
            this.params = params;
            this.processBuilder = command.toBuilder()
                    .exec(getPaths().getFFmpeg().getAbsolutePath())
                    .in("-hide_banner") // suppress printing banner (copyright, build options, library versions)
                    .build()
                    .toProcessBuilder();
        }

        public String getSourceKey() {
            return command.getSource().key;
        }

        public Session executeAsync() {
            executor.execute(this);
            session.waitForProcess();
            return session;
        }

        public void execute() {
            TimeAndSuccessLogUtil.runLoggingTimeAndSuccess(this::doExecute, "FFmpeg command", logger);
        }

        private void doExecute() {
            try {
                Process process;
                try {
                    process = processBuilder.start();
                    startTimeO = Option.of(Instant.now());
                    session.setProcess(process);
                    logger.info("Started {}", getCommandStr());
                } catch (IOException ex) {
                    logger.info("Error while starting {}", getCommandStr(), ex);
                    session.setStartProcessException(ex);
                    throw ex;
                }

                CommonThreadPoolHolder.runAsync(() -> handleStdout(process.getInputStream()));
                CommonThreadPoolHolder.runAsync(() -> handleStderr(process.getErrorStream()));

                try {
                    waitFor(process);
                    if (process.exitValue() != 0) {
                        throw convertStderrToException(command.getSource(), StringUtils.join(stderrContents, "\n"));
                    }
                } catch (RuntimeException ex) {
                    logger.error("Error while executing {}", getCommandStr(), ex);
                    throw ex;
                }

                session.setExecutionFinished();
            } catch(IOException | RuntimeException ex) {
                session.setExecutionException(ex);
                handleExecutionError(ex);
                throw ExceptionUtils.translate(ex);
            } finally {
                command.getTarget().close();
            }
        }

        protected void waitFor(Process process) {
            WaitInCycle<Boolean> cycle = new WaitInCycle<Boolean>(WAIT_FOR_SLEEP_DURATION, params.getTranscodingTimeout())
                    .repeatWhile(exited -> !exited);
            boolean ffmpegExited = cycle.call(duration -> {
                Duration elapsed = cycle.elapsed();
                if (lastStatsO.isMatch(stats -> stats.elapsed.equals(Duration.ZERO)
                        && elapsed.isLongerThan(params.getEncodingStartTimeout())))
                {
                    throw new IllegalStateException("Encoding is not started in " + elapsed);
                }

                try {
                    return process.waitFor(duration.getMillis(), TimeUnit.MILLISECONDS);
                } catch (InterruptedException e) {
                    throw ExceptionUtils.translate(e);
                }
            });

            if (!ffmpegExited) {
                session.destroyForciblyQuietly();
                throw new IllegalStateException("FFmpeg execution timed out after " + cycle.elapsed());
            }
        }

        public Option<Duration> estimateLeftOverDurationO() {
            if (session.isFinished()) {
                return Option.of(Duration.ZERO);
            } else if (!lastStatsO.isPresent() || !command.getDurationO().isPresent()) {
                return Option.empty();
            } else {
                FFmpegStats stats = lastStatsO.get();
                return DurationEstimator.consO(
                        startTimeO.get(), stats.createdAt, stats.elapsed, command.getDurationO().get().toJodaDuration())
                        .map(DurationEstimator::estimateRealLeftOverDuration);
            }
        }

        private void handleStdout(InputStream inputStream) {
            getInputStreamConsumer()
                    .accept(inputStream);
        }

        private Consumer<InputStream> getInputStreamConsumer() {
            return command.getPipeConsumerO()
                    .getOrElse(
                            () -> in -> new InputStreamX(in)
                                    .readToDevNull()
                    );
        }

        private void handleStderr(InputStream errorStream) {
            InputStreamSourceUtils.wrap(errorStream)
                    .asReaderTool()
                    .forEachLine(this::handleStderrLine);
        }

        private void handleStderrLine(String line) {
            if (line.length() > MAX_STDERR_LINE_SIZE) {
                logger.warn("Cutting too large stderr line: {} > {}", line.length(), MAX_STDERR_LINE_SIZE);
                // TODO: do not read whole line in memory in the first place
                line = line.substring(0, MAX_STDERR_LINE_SIZE);
            }

            if (params.isOutputLoggingEnabled()) {
                ffmpegLogger.trace(line);
            }

            Option<FFmpegStats> stats = FFmpegStats.parseStderr(line);
            if (stats.isPresent()) {
                lastStatsO = Option.of(stats.get());
                // discard everything before stats - it is definitely not an error
                stderrContents.clear();
                statsHandler.handle(stats.get());
            } else {
                stderrContents.add(line);
            }
        }

        private void handleExecutionError(Exception ex) {
            try {
                errorHandler.accept(ExceptionUtils.translate(ex));
            } catch (Throwable e) {
                logger.warn("Got error while handling FFmpeg error", e);
            }
        }

        private String getCommandStr() {
            return FFmpeg.toString(processBuilder);
        }
    }

    private static FFmpegExecException convertStderrToException(FFmpegSource source, String stderr) {
        if (stderr.contains("HTTP error 404 Not Found")) {
            return new SourceNotFoundFFmpegException(source);
        } else if (stderr.contains("Invalid data found when processing input")) {
            return new UnprocessableSourceFFmpegException(source);
        } else {
            return new FFmpegExecException(source, stderr);
        }
    }

    public static class FFmpegException extends RuntimeException {
        public FFmpegException(String message) {
            super(message);
        }

        public FFmpegException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class FFmpegExecException extends FFmpegException {
        public final FFmpegSource source;

        public FFmpegExecException(FFmpegSource source, String message) {
            super(message, FFmpegSourceExceptionRegistry.INSTANCE.getExceptionO(source).getOrNull());
            this.source = source;
        }
    }

    public static class SourceNotFoundFFmpegException extends FFmpegExecException implements HlsErrorSource {
        public SourceNotFoundFFmpegException(FFmpegSource source) {
            super(source, "Source not found");
        }

        @Override
        public HlsError getHlsError() {
            return new HlsError(HttpStatus.SC_404_NOT_FOUND, "Source not found");
        }
    }

    public static class UnprocessableSourceFFmpegException extends FFmpegExecException implements HlsErrorSource {
        public UnprocessableSourceFFmpegException(FFmpegSource source) {
            super(source, "Invalid data found when processing input");
        }

        @Override
        public HlsError getHlsError() {
            return new HlsError(HttpStatus.SC_422_UNPROCESSABLE_ENTITY,
                    "Could not process resource - probably not a media file");
        }
    }
}
