package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import java.util.function.Consumer;
import java.util.function.Function;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFmpegCommandBuilder {
    private Option<String> execO = Option.empty();

    protected final ProcessCommandBuilder inParams;

    private FFmpegSource source;

    protected final ProcessCommandBuilder outParams;

    private FFmpegOutput out;

    private Option<MediaTime> durationO;

    public FFmpegCommandBuilder(FFmpegSource source, FFmpegOutput out) {
        this(Option.empty(), new ProcessCommandBuilder(), source, new ProcessCommandBuilder(), out, Option.empty());
    }

    FFmpegCommandBuilder(FFmpegCommand command) {
        this(command.execO, command.inParams.toBuilder(), command.source, command.outParams.toBuilder(), command.out,
                command.durationO);
    }

    private FFmpegCommandBuilder(Option<String> execO, ProcessCommandBuilder inParams, FFmpegSource source,
            ProcessCommandBuilder outParams, FFmpegOutput out, Option<MediaTime> durationO)
    {
        this.execO = execO;
        this.inParams = inParams;
        this.source = source;
        this.outParams = outParams;
        this.out = out;
        this.durationO = durationO;
    }

    public FFmpegCommandBuilder add(FFmpegCommandBuilder another) {
        inParams.add(another.inParams);
        outParams.add(another.outParams);
        source = another.source;
        out = another.out;
        return this;
    }

    public FFmpegCommandBuilder.WithCondition applyIf(Consumer<FFmpegCommandBuilder> consumer, boolean condition) {
        if (condition) {
            consumer.accept(this);
        }
        return new WithCondition(this, condition);
    }

    public FFmpegCommandBuilder exec(String path) {
        this.execO = Option.of(path);
        return this;
    }

    public FFmpegCommandBuilder inO(Option<?> paramO) {
        return in(builder -> builder.addO(paramO));
    }

    public FFmpegCommandBuilder inO(String param, Option<?> value) {
        return in(builder -> builder.addO(param, value));
    }

    public FFmpegCommandBuilder in(String param) {
        return in(builder -> builder.add(param));
    }

    public FFmpegCommandBuilder in(String param, Object value) {
        return in(builder -> builder.add(param, value));
    }

    private FFmpegCommandBuilder in(Consumer<ProcessCommandBuilder> consumer) {
        return transform(inParams, consumer);
    }

    public FFmpegCommandBuilder.WithCondition inIf(String param, boolean condition) {
        return inIf(builder -> builder.addIf(param, condition));
    }

    public FFmpegCommandBuilder.WithCondition inIf(String param, Object value, boolean condition) {
        return inIf(builder -> builder.addIf(param, value, condition));
    }

    private FFmpegCommandBuilder.WithCondition inIf(
            Function<ProcessCommandBuilder, ProcessCommandBuilder.WithCondition> function)
    {
        return transformIf(inParams, function);
    }

    public FFmpegCommandBuilder outO(Option<?> paramO) {
        return out(builder -> builder.addO(paramO));
    }

    public FFmpegCommandBuilder outO(String param, Option<?> value) {
        return out(builder -> builder.addO(param, value));
    }

    public FFmpegCommandBuilder out(String param) {
        return out(builder -> builder.add(param));
    }

    public FFmpegCommandBuilder out(String param, Object value) {
        return out(builder -> builder.add(param, value));
    }

    private FFmpegCommandBuilder out(Consumer<ProcessCommandBuilder> consumer) {
        return transform(outParams, consumer);
    }

    public FFmpegCommandBuilder.WithCondition outIf(String param, boolean condition) {
        return outIf(builder -> builder.addIf(param, condition));
    }

    public FFmpegCommandBuilder.WithCondition outIf(String param, Object value, boolean condition) {
        return outIf(builder -> builder.addIf(param, value, condition));
    }

    private FFmpegCommandBuilder.WithCondition outIf(
            Function<ProcessCommandBuilder, ProcessCommandBuilder.WithCondition> function)
    {
        return transformIf(outParams, function);
    }

    private FFmpegCommandBuilder transform(ProcessCommandBuilder builder, Consumer<ProcessCommandBuilder> consumer) {
        consumer.accept(builder);
        return this;
    }

    private FFmpegCommandBuilder.WithCondition transformIf(ProcessCommandBuilder builder,
            Function<ProcessCommandBuilder, ProcessCommandBuilder.WithCondition> consumer)
    {
        return new WithCondition(this, consumer.apply(builder).getCondition());
    }

    public ListF<String> toStringList() {
        return build()
                .toStringList();
    }

    public FFmpegCommand build() {
        return new FFmpegCommand(execO, inParams.build(), source, outParams.build(), out, durationO);
    }

    public static class WithCondition extends FFmpegCommandBuilder {
        private final boolean condition;

        private WithCondition(FFmpegCommandBuilder proto, boolean condition) {
            super(proto.execO, proto.inParams, proto.source, proto.outParams, proto.out, proto.durationO);
            this.condition = condition;
        }

        public FFmpegCommandBuilder elseIn(String param) {
            return orElse(inParams, param);
        }

        public FFmpegCommandBuilder elseOut(String param) {
            return orElse(outParams, param);
        }

        public FFmpegCommandBuilder elseIn(String param, Object value) {
            return orElse(inParams, param, value);
        }

        public FFmpegCommandBuilder elseOut(String param, Object value) {
            return orElse(outParams, param, value);
        }

        private FFmpegCommandBuilder orElse(ProcessCommandBuilder params, String param) {
            if (!condition) {
                params.add(param);
            }
            return this;
        }

        private FFmpegCommandBuilder orElse(ProcessCommandBuilder params, String param, Object value) {
            if (!condition) {
                params.add(param, value);
            }
            return this;
        }
    }
}
