package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.util.threadlocal.ReplicateThreadLocalsUtil;
import ru.yandex.chemodan.videostreaming.framework.web.HlsErrorSource;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFmpegExecutor {
    private static final Logger logger = LoggerFactory.getLogger(FFmpegExecutor.class);

    private static final DynamicProperty<Boolean> limitTranscodingToOnePerSourceEnabled =
            DynamicProperty.cons("streaming-limit-transcoding-to-one-per-source-enabled",false);

    private static final DynamicProperty<Integer> limitTranscodingPerSourcePerDc =
            DynamicProperty.cons("streaming-limit-transcoding-per-source-per-dc",0);

    private final ThreadPoolExecutor executor;

    private final TranscodingInventory transcodingInventory;

    final MapF<UUID, FFmpeg.Execution> activeExecutions = Cf.x(new ConcurrentHashMap<>());

    public FFmpegExecutor(TranscodingInventory transcodingInventory) {
        this.executor = new ThreadPoolExecutor(8, 8, 0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>()
        );
        this.transcodingInventory = transcodingInventory;
    }

    public synchronized void execute(FFmpeg.Execution execution) {
        increasePoolIfNeeded(execution.params);
        checkFFmpegExecutionLimit(execution);
        executor.execute(ReplicateThreadLocalsUtil.attachTo(() -> doExecute(execution)));
    }

    void doExecute(FFmpeg.Execution execution) {
        UUID key = UUID.randomUUID();
        transcodingInventory.add(execution.getSourceKey(), key);
        activeExecutions.put(key, execution);
        try {
            execution.execute();
        } finally {
            activeExecutions.remove(key);
            transcodingInventory.remove(execution.getSourceKey(), key);
            logger.info("Completed execution. Active count: {}", activeExecutions.size());
        }
    }

    private void checkFFmpegExecutionLimit(FFmpeg.Execution execution) {
        String sourceKey = execution.getSourceKey();

        int sourceTranscodingPerDcLimit = limitTranscodingPerSourcePerDc.get();
        if (sourceTranscodingPerDcLimit > 0 && transcodingInventory.getCount(sourceKey) >= sourceTranscodingPerDcLimit) {
            logger.info("Exceeded per DC transcoding limit = {} for source = {}", sourceTranscodingPerDcLimit, sourceKey);
            throw new FFmpegExecLimitException("Exceeded per source per DC transcoding limit");
        }

        if (limitTranscodingToOnePerSourceEnabled.get() && hasActiveSourceKey(sourceKey)) {
            logger.info("Only one simultaneous transcoding is allowed per source = {}", sourceKey);
            throw new FFmpegExecLimitException("Only one simultaneous transcoding per source is allowed");
        }

        if (executionDenied(execution.params)) {
            logger.info("Exceeded FFmpeg transcoding limit");
            throw new FFmpegExecLimitException("FFmpeg execution limit reached");
        }
    }

    private boolean hasActiveSourceKey(String sourceKey) {
        return activeExecutions.values()
                .exists(activeExecution -> activeExecution.getSourceKey().equals(sourceKey));
    }

    private boolean executionDenied(FFmpegParams params) {
        return executor.getActiveCount() >= executor.getMaximumPoolSize()
                && getAlmostDoneCount(params) <= executor.getQueue().size();
    }

    private int getAlmostDoneCount(FFmpegParams params) {
        int result = 0;
        for(FFmpeg.Execution exec : activeExecutions.values()) {
            Option<Duration> leftOverDurationO = exec.estimateLeftOverDurationO();
            if (leftOverDurationO.isMatch(duration -> duration.isShorterThan(params.getMinLeftOverDuration()))) {
                logger.info("Got left over duration = {}", leftOverDurationO.get());
                result++;
            }
        }
        return result;
    }

    private void increasePoolIfNeeded(FFmpegParams params) {
        if (params.getMaxProcCountO().isMatch(maxSize -> executor.getMaximumPoolSize() != maxSize)) {
            if (executor.getCorePoolSize() > params.getMaxProcCountO().get()) {
                executor.setCorePoolSize(params.getMaxProcCountO().get());
            }
            executor.setMaximumPoolSize(params.getMaxProcCountO().get());
        }
    }

    public static class FFmpegExecLimitException extends FFmpeg.FFmpegException implements HlsErrorSource {
        public FFmpegExecLimitException(String message) {
            super(message);
        }

        @Override
        public HlsError getHlsError() {
            return new HlsError(HttpStatus.SC_429_TOO_MANY_REQUESTS, getMessage());
        }
    }
}
