package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import java.io.InputStream;
import java.net.URI;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public abstract class FFmpegOutput extends DefaultObject implements AutoCloseable {
    protected abstract String getStrValue();

    public static FFmpegOutput raw(String strValue, Runnable closer) {
        return new RawOutput(strValue, closer);
    }

    public static FFmpegOutput uri(String uri) {
        return uri(new UriBuilder(uri).build());
    }

    public static FFmpegOutput uri(URI uri) {
        return new UriOutput(uri);
    }

    public static FFmpegOutput path(File2 file) {
        return new FileOutput(file);
    }

    @SuppressWarnings("unused")
    public static FFmpegOutput pipe(Consumer<InputStream> consumer) {
        return new PipeOutput(consumer);
    }

    public Option<Consumer<InputStream>> getPipeConsumerO() {
        return Option.empty();
    }

    @Override
    public String toString() {
        return getStrValue();
    }

    @Override
    public void close() {
        // do nothing by default - override in subclasses
    }

    private static class FileOutput extends FFmpegOutput {
        private final File2 file;

        private FileOutput(File2 file) {
            this.file = file;
        }

        @Override
        protected String getStrValue() {
            return file.getAbsolutePath();
        }
    }

    private static class UriOutput extends FFmpegOutput {
        private final URI uri;

        private UriOutput(URI uri) {
            this.uri = uri;
        }

        @Override
        protected String getStrValue() {
            return uri.toASCIIString();
        }
    }

    private static class RawOutput extends FFmpegOutput {
        private final String value;

        private final Runnable closer;

        private RawOutput(String value, Runnable closer) {
            this.value = value;
            this.closer = closer;
        }

        @Override
        protected String getStrValue() {
            return value;
        }

        @Override
        public void close() {
            closer.run();
        }
    }

    private static class PipeOutput extends FFmpegOutput {
        private final Consumer<InputStream> pipeConsumer;

        private PipeOutput(Consumer<InputStream> pipeConsumer) {
            this.pipeConsumer = pipeConsumer;
        }

        @Override
        protected String getStrValue() {
            return "pipe";
        }

        @Override
        public Option<Consumer<InputStream>> getPipeConsumerO() {
            return Option.of(pipeConsumer);
        }
    }
}
