package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author vavinov
 * @author Dmitriy Amelin (lemeh)
*/
public class FFmpegStats extends DefaultObject {
    private static final Pattern2 STDERR_LINE_PATTERN =
            Pattern2.compile("" +
                    "frame= *([0-9]+)"
                    + " fps= *([0-9\\.]+)"
                    + " q= *([^ ]+)"
                    + " size=(?: *([0-9]+)kB|N/A)"
                    + " time= *([0-9:.]+)"
                    + ".+speed=([0-9\\.]+)x"
            );

    public transient final Instant createdAt = Instant.now();

    public final long frames;
    public final double fps;
    public final Duration elapsed;
    public final Option<Long> bytes;
    public final double speed;

    public FFmpegStats(long frames, double fps, Duration elapsed, double speed, long bytes) {
        this(frames, fps, elapsed, speed, Option.of(bytes));
    }

    public FFmpegStats(long frames, double fps, Duration elapsed, double speed, Option<Long> bytes) {
        this.frames = frames;
        this.fps = fps;
        this.elapsed = elapsed;
        this.bytes = bytes;
        this.speed = speed;
    }

    public static Option<FFmpegStats> parseStderr(String line) {
        line = StringUtils.stripEnd(line, "\r")
                .substring(line.lastIndexOf('\r') + 1);

        Matcher2 m = STDERR_LINE_PATTERN.matcher2(line);
        if (!m.find()) {
            return Option.empty();
        }

        try {
            long frames = Long.parseLong(m.group(1).get());
            double fps = Double.parseDouble(m.group(2).get());
            Option<Long> bytes = m.group(4)
                    .map(s -> Long.valueOf(s) * 1024);
            Duration millis = FFmpegTimeDurationUtil.parse(m.group(5).get()).toJodaDuration();
            double speed = Double.valueOf(m.group(6).get());
            return Option.of(new FFmpegStats(frames, fps, millis, speed, bytes));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }
}
