package ru.yandex.chemodan.videostreaming.framework.ffmpeg;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.cache.processor.EntryProcessorException;
import javax.cache.processor.MutableEntry;

import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.cache.CacheEntryProcessor;
import org.apache.ignite.configuration.CacheConfiguration;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.commune.dynproperties.DynamicProperty;

public class TranscodingInventory {
    private static final DynamicProperty<Boolean> transcodingInventoryEnabled =
            DynamicProperty.cons("streaming-transcoding-inventory-enabled", false);

    private final Ignite ignite;

    private final CacheConfiguration<String, Transcodings> transcodingsCacheConfig;

    private final FFmpegParams ffmpegParams;

    public TranscodingInventory(
            Ignite ignite,
            CacheConfiguration<String, Transcodings> transcodingsCacheConfig,
            FFmpegParams ffmpegParams
    ) {
        this.ignite = ignite;
        this.transcodingsCacheConfig = transcodingsCacheConfig;
        this.ffmpegParams = ffmpegParams;
    }

    public int getCount(String sourceKey) {
        return transcodingInventoryEnabled.get()
                ? getTranscodingsCache().invoke(sourceKey, new CountTranscodings())
                : 0;
    }

    public void add(String sourceKey, UUID id) {
        if (transcodingInventoryEnabled.get()) {
            getTranscodingsCache().invoke(sourceKey, new AddTranscoding(ffmpegParams.getTranscodingTimeout(), id));
        }
    }

    public void remove(String sourceKey, UUID id) {
        if (transcodingInventoryEnabled.get()) {
            getTranscodingsCache().invoke(sourceKey, new RemoveTranscoding(id));
        }
    }

    public IgniteCache<String, Transcodings> getTranscodingsCache() {
        return ignite.getOrCreateCache(transcodingsCacheConfig);
    }

    public static class CountTranscodings implements CacheEntryProcessor<String, Transcodings, Integer> {
        @Override
        public Integer process(MutableEntry<String, Transcodings> entry, Object... arguments) throws EntryProcessorException {
            Transcodings transcodings = entry.exists() ? entry.getValue() : null;
            if (transcodings != null) {
                transcodings.evict();
                return transcodings.size();
            } else {
                return 0;
            }
        }
    }

    public static class AddTranscoding implements CacheEntryProcessor<String, Transcodings, Void> {
        Duration ttl;

        UUID id;

        AddTranscoding(Duration ttl, UUID id) {
            this.ttl = ttl;
            this.id = id;
        }

        @Override
        public Void process(MutableEntry<String, Transcodings> entry, Object... arguments) throws EntryProcessorException {
            if (!entry.exists()) {
                entry.setValue(new Transcodings(ttl));
            }

            entry.getValue().add(id);
            return null;
        }
    }

    public static class RemoveTranscoding implements CacheEntryProcessor<String, Transcodings, Boolean> {
        UUID id;

        RemoveTranscoding(UUID id) {
            this.id = id;
        }

        @Override
        public Boolean process(MutableEntry<String, Transcodings> entry, Object... arguments) throws EntryProcessorException {
            if (!entry.exists()) {
                return false;
            }

            entry.getValue().remove(id);
            return true;
        }
    }

    public static class Transcodings implements Serializable {
        Duration ttl;

        Map<UUID, Instant> idToStartTimeMap = new HashMap<>();

        Transcodings(Duration ttl) {
            this.ttl = ttl;
        }

        void add(UUID id) {
            idToStartTimeMap.put(id, Instant.now());
        }

        void remove(UUID id) {
            idToStartTimeMap.remove(id);
        }

        void evict() {
            idToStartTimeMap = idToStartTimeMap.entrySet()
                    .stream()
                    .filter(idToStartTime ->
                            new Duration(idToStartTime.getValue(), Instant.now())
                                    .isShorterThan(ttl)
                    )
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        }

        public boolean contains(UUID id) {
            return idToStartTimeMap.containsKey(id);
        }

        public int size() {
            return idToStartTimeMap.size();
        }

        @Override
        public String toString() {
            return "transcodings-size=" + size();
        }
    }
}
