package ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe;

import java.io.IOException;
import java.io.InputStream;
import java.util.function.Consumer;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.units.AspectRatio;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.media.units.Rotation;
import ru.yandex.chemodan.videostreaming.framework.media.units.SampleFrequency;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFprobeInfoTest {
    private static final String TEST_JSON_FILENAME = "guardians_of_the_galaxy_vol2_2017_tlr1-DWEU.mkv.json";

    private static final String TEST_JSON_N2_FILENAME = "2018-03-31_18-54-12.MOV.json";

    private static final String MISSING_CODEC_TYPE_FILENAME = "av_other_wo_codec_type.json";

    private static final String BROKEN_JSON_FILENAME = "broken_json_due_ff_bug.json";

    @Test
    public void parse() {
        FFprobeFormat format = FFprobeFormat.builder()
                .name("matroska,webm")
                .duration(MediaTime.seconds(94).plus(262000))
                .bitRate(new BitRate(13718476))
                .probeScore(100)
                .tags(new FFprobeFormat.Tags(Option.of("2017-02-28T20:33:17.000000Z")))
                .build();
        FFprobeStream.Video videoStream = FFprobeStream.Video.builder()
                .index(0)
                .codecName("h264")
                .dimension(new Dimension(1920, 1080))
                .displayAspectRatio(new AspectRatio(16, 9))
                .rFrameRate(new FrameRate(24000, 1001))
                .avgFrameRate(new FrameRate(24000, 1001))
                .build();
        FFprobeStream.Audio audioStream1 = FFprobeStream.Audio.builder()
                .index(1)
                .codecName("dts")
                .sampleRate(new SampleFrequency(48000))
                .channels(6)
                .channelLayout("5.1(side)")
                .build();
        FFprobeStream.Audio audioStream2 = FFprobeStream.Audio.builder()
                .index(2)
                .codecName("ac3")
                .bitRate(new BitRate(640000))
                .sampleRate(new SampleFrequency(48000))
                .channels(6)
                .channelLayout("5.1(side)")
                .build();
        FFprobeInfo expected = new FFprobeInfo(format, Cf.list(videoStream, audioStream1, audioStream2));
        withTestFile(TEST_JSON_FILENAME, in -> Assert.equals(expected, FFprobeInfo.parse(in.readBytes())));
    }

    @Test
    public void parse2() {
        FFprobeFormat format = FFprobeFormat.builder()
                .name("mov,mp4,m4a,3gp,3g2,mj2")
                .duration(MediaTime.seconds(20).plus(154921))
                .bitRate(new BitRate(7769720))
                .probeScore(100)
                .tags(new FFprobeFormat.Tags(Option.of("2018-03-31T15:54:12.000000Z")))
                .build();
        FFprobeStream.Video videoStream = FFprobeStream.Video.builder()
                .index(0)
                .codecName("hevc")
                .dimension(new Dimension(1920, 1080))
                .displayAspectRatio(new AspectRatio(0, 1))
                .rFrameRate(new FrameRate(30, 1))
                .avgFrameRate(new FrameRate(363000, 12097))
                .bitRate(new BitRate(7672926))
                .tags(new FFprobeStream.Video.Tags(Option.of(new Rotation(90))))
                .build();
        FFprobeStream.Audio audioStream = FFprobeStream.Audio.builder()
                .index(1)
                .codecName("aac")
                .sampleRate(new SampleFrequency(44100))
                .channels(1)
                .channelLayout("mono")
                .bitRate(new BitRate(88854))
                .build();
        FFprobeStream.Other data1 = new FFprobeStream.Other();
        FFprobeStream.Other data2 = new FFprobeStream.Other();
        FFprobeInfo expected = new FFprobeInfo(format, videoStream, audioStream, data1, data2);
        withTestFile(TEST_JSON_N2_FILENAME, in -> Assert.equals(expected, FFprobeInfo.parse(in.readBytes())));
    }

    @Test
    public void parseWhenStreamCodecTypeIsMissing() {
        assertParsedClasses(MISSING_CODEC_TYPE_FILENAME, FFprobeStream.Video.class, FFprobeStream.Audio.class,
                FFprobeStream.Other.class);
    }

    @Test
    public void parseBrokenJson() {
        assertParsedClasses(BROKEN_JSON_FILENAME, FFprobeStream.Video.class, FFprobeStream.Audio.class,
                FFprobeStream.Audio.class, FFprobeStream.Audio.class, FFprobeStream.Audio.class);
    }

    @SafeVarargs
    private final void assertParsedClasses(String filename, Class<? extends FFprobeStream>... expectedClasses) {
        withTestFile(filename, in -> Assert.equals(
                Cf.list(expectedClasses),
                FFprobeInfo.parse(in.readBytes())
                        .getStreams()
                        .map(FFprobeStream::getClass)
        ));
    }

    private static void withTestFile(String filename, Consumer<InputStreamX> consumer) {
        try (InputStream in = FFprobeInfoTest.class.getResourceAsStream(filename)) {
            consumer.accept(new InputStreamX(in));
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
