package ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe;

import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.NonNull;
import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.chemodan.videostreaming.framework.media.units.AspectRatio;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.media.units.Rotation;
import ru.yandex.chemodan.videostreaming.framework.media.units.SampleFrequency;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.image.Dimension;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public abstract class FFprobeStream {
    private static final MapF<String, BenderParser<? extends FFprobeStream>> streamParsers = Cf.map(
            "video", BenderUtil.consParser(Video.class),
            "audio", BenderUtil.consParser(Audio.class)
    );

    private static final BenderParser<Other> otherParser = BenderUtil.consParser(Other.class);

    public static JsonFieldLevelUnmarshaller unmarshaller = (node, context) -> ParseResult.result(parse(node));

    private static FFprobeStream parse(BenderJsonNode node) {
        return node.getField("codec_type")
                    .map(BenderJsonNode::getString)
                    .filterMap(streamParsers::getO)
                    .getOrElse(otherParser)
                    .parseJson(node);
    }

    public abstract int getIndex();

    public abstract Option<String> getCodecName();

    public abstract Option<BitRate> getBitRate();

    public Option<MediaTime> getDuration() {
        return Option.empty();
    }

    @Value
    @Builder(builderClassName = "Builder", toBuilder = true)
    @EqualsAndHashCode(callSuper = false)
    @BenderBindAllFields
    public static class Video extends FFprobeStream {
        int index;

        @NonNull
        Option<String> codecName;

        @NonNull
        Option<BitRate> bitRate;

        @BenderFlatten
        Dimension dimension;

        @NonNull
        Option<AspectRatio> displayAspectRatio;

        @NonNull
        FrameRate rFrameRate;

        @NonNull
        FrameRate avgFrameRate;

        @NonNull
        Option<MediaTime> duration;

        @NonNull
        Option<Tags> tags;

        @Value
        @BenderBindAllFields
        @lombok.Builder(builderClassName = "Builder", toBuilder = true)
        public static class Tags {
            @NonNull
            Option<Rotation> rotate;

            public static class Builder {
                @SuppressWarnings("unused") // default value
                Option<Rotation> rotate = Option.empty();
            }
        }

        @SuppressWarnings("unused")
        public static class Builder {
            private Option<String> codecName = Option.empty();

            private Option<BitRate> bitRate = Option.empty();

            private Option<AspectRatio> displayAspectRatio = Option.empty();

            private Option<Tags> tags = Option.empty();

            private Option<MediaTime> duration = Option.empty();

            public Builder codecName(String codecName) {
                return codecName(Option.of(codecName));
            }

            public Builder codecName(Option<String> codecName) {
                this.codecName = codecName;
                return this;
            }

            public Builder bitRate(BitRate bitRate) {
                return bitRate(Option.of(bitRate));
            }

            public Builder bitRate(Option<BitRate> bitRate) {
                this.bitRate = bitRate;
                return this;
            }

            public Builder displayAspectRatio(AspectRatio displayAspectRatio) {
                return displayAspectRatio(Option.of(displayAspectRatio));
            }

            public Builder displayAspectRatio(Option<AspectRatio> displayAspectRatio) {
                this.displayAspectRatio = displayAspectRatio;
                return this;
            }

            public Builder duration(MediaTime duration) {
                return duration(Option.of(duration));
            }

            public Builder duration(Option<MediaTime> duration) {
                this.duration = duration;
                return this;
            }

            public Builder tags(Tags tags) {
                return tags(Option.of(tags));
            }

            public Builder tags(Option<Tags> tags) {
                this.tags = tags;
                return this;
            }
        }
    }

    @Value
    @Builder(builderClassName = "Builder", toBuilder = true)
    @EqualsAndHashCode(callSuper = false)
    @BenderBindAllFields
    public static class Audio extends FFprobeStream {
        int index;

        @NonNull
        Option<String> codecName;

        @NonNull
        Option<BitRate> bitRate;

        @NonNull
        SampleFrequency sampleRate;

        int channels;

        @NonNull
        Option<String> channelLayout;

        @NonNull
        Option<MediaTime> duration;

        @SuppressWarnings("unused")
        public static class Builder {
            private Option<String> codecName = Option.empty();

            private Option<BitRate> bitRate = Option.empty();

            private Option<String> channelLayout = Option.empty();

            private Option<MediaTime> duration = Option.empty();

            public Builder codecName(String codecName) {
                return codecName(Option.of(codecName));
            }

            public Builder codecName(Option<String> codecName) {
                this.codecName = codecName;
                return this;
            }

            public Builder bitRate(BitRate bitRate) {
                return bitRate(Option.of(bitRate));
            }

            public Builder bitRate(Option<BitRate> bitRate) {
                this.bitRate = bitRate;
                return this;
            }

            public Builder channelLayout(String channelLayout) {
                return channelLayout(Option.of(channelLayout));
            }

            public Builder channelLayout(Option<String> channelLayout) {
                this.channelLayout = channelLayout;
                return this;
            }

            public Builder duration(MediaTime duration) {
                return duration(Option.of(duration));
            }

            public Builder duration(Option<MediaTime> duration) {
                this.duration = duration;
                return this;
            }
        }
    }

    @Value
    @EqualsAndHashCode(callSuper = false)
    @Bendable
    public static class Other extends FFprobeStream {
        final int index = -1;

        final Option<String> codecName = Option.empty();

        final Option<BitRate> bitRate = Option.empty();
    }
}
