package ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.analysis;

import java.math.RoundingMode;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.misc.io.file.File2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFprobeOutputAnalyzer {
    private static final ListF<String> CODECS_OF_INTEREST = Cf.list("aic", "png", "jpeg2000");

    private final ValueCounts<FrameRate> rFrameRateCounts = ValueCounts.frameRate();

    private final ValueCounts<FrameRate> avgFrameRateCounts = ValueCounts.frameRate();

    private final ValueCounts<FrameRate> matchingFrameRateCounts = ValueCounts.frameRate();

    private final AtomicInteger allCounter = new AtomicInteger(0);

    private final AtomicInteger nonEqualFpsCounter = new AtomicInteger(0);

    private final AtomicInteger videoCounter = new AtomicInteger(0);

    private final ValueCounts<String> videoCodecCounts = new ValueCounts<>();

    private void process(Tuple2<FFprobeInfo, File2> info) {
        process(info.get1());
    }

    private void process(FFprobeInfo info) {
        info.getVideoStreamO()
                .ifPresent(video -> {
                    video.getCodecName()
                            .ifPresent(videoCodecCounts::add);

                    avgFrameRateCounts.add(video.getAvgFrameRate());
                    rFrameRateCounts.add(video.getRFrameRate());
                    if (!video.getAvgFrameRate().equals(video.getRFrameRate())) {
                        nonEqualFpsCounter.incrementAndGet();
                    } else {
                        matchingFrameRateCounts.add(video.getAvgFrameRate());
                    }
                    videoCounter.incrementAndGet();
                });
    }

    private void printStats() {
        avgFrameRateCounts.printStats("avg frame rate");
        rFrameRateCounts.printStats("r-frame rate");
        matchingFrameRateCounts.printStats("matching frame rate");
        avgFrameRateCounts.printUniqueCount("avg frame rate");
        rFrameRateCounts.printUniqueCount("r-frame rate");
        matchingFrameRateCounts.printUniqueCount("matching frame rate");

        videoCodecCounts.printStats("video codec");
        videoCodecCounts.printUniqueCount("video codec");
        System.out.println();

        printCounter("Non-equal FPS", nonEqualFpsCounter, videoCounter);
        printCounter("Video", videoCounter, allCounter);
    }

    private void printCounter(String message, AtomicInteger counter, AtomicInteger totalCounter) {
        System.out.println(String.format("%s: %.2f%% (%d out of %d)", message,
                counter.doubleValue() / totalCounter.doubleValue() * 100.0,
                counter.intValue(), totalCounter.intValue())
        );
    }

    private static class ValueCounts<T> {
        final MapF<T, AtomicInteger> counts = Cf.wrap(new ConcurrentHashMap<>());

        final int maxCount;

        final Function<T, String> toStringF;

        ValueCounts() {
            this(Integer.MAX_VALUE, Object::toString);
        }

        ValueCounts(int maxCount, Function<T, String> toStringF) {
            this.maxCount = maxCount;
            this.toStringF = toStringF;
        }

        static ValueCounts<FrameRate> frameRate() {
            return new ValueCounts<>(100, frameRate -> "" + frameRate + ", "
                    + frameRate.toBigDecimal().setScale(3, RoundingMode.HALF_UP));
        }

        void add(T value) {
            counts.computeIfAbsent(value, fr -> new AtomicInteger());
            counts.getTs(value)
                    .incrementAndGet();
        }

        void printStats(String name) {
            System.out.println(name);
            counts.entries()
                    .sortedBy2Desc(AtomicInteger::get)
                    .take(maxCount)
                    .reverse()
                    .forEach(this::printValueCount);
            System.out.println();
        }

        private void printValueCount(Tuple2<T, AtomicInteger> valueCount) {
            T value = valueCount.get1();
            System.out.println(toStringF.apply(value) + ": " + valueCount.get2());
        }

        void printUniqueCount(String name) {
            System.out.println("Unique " + name + " count = " + counts.size());
        }
    }

    public static void main(String[] args) {
        FFprobeOutputAnalyzer processor = new FFprobeOutputAnalyzer();
        new MultiThreadedAnalyzer(processor::process, 30)
                .run();
        processor.printStats();
    }
}
