package ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.analysis;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeInfo;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.file.File2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MultiThreadedAnalyzer {
    final ExecutorService executor;

    final Consumer<Tuple2<FFprobeInfo, File2>> consumer;

    final Path path = Paths.get("/Users/lemeh/Downloads/streaming/ffprobe/");

    final PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher("glob:**/*.par");

    final AtomicInteger fileCounter = new AtomicInteger();

    public MultiThreadedAnalyzer(Consumer<Tuple2<FFprobeInfo, File2>> consumer, int nThreads) {
        this.executor = Executors.newFixedThreadPool(nThreads);
        this.consumer = consumer;
    }

    public void run() {
        try {
            runUnsafe();
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
        shutdown();
    }

    private void runUnsafe() throws IOException {
        Files.walkFileTree(path, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) {
                if (pathMatcher.matches(path)) {
                    processAsync(path);
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void processAsync(Path path) {
        executor.submit(() -> process(path));
    }

    private void process(Path path) {
        File2 file = new File2(path.toFile());
        int number = fileCounter.incrementAndGet();
        String fileId = String.format("file#%d: %s", number, file);

        try {
            byte[] json = file.readBytes();
            consumer.accept(new Tuple2<>(FFprobeInfo.parse(json), file));
        } catch (RuntimeException ex) {
            System.out.println("Error while processing " + fileId);
            ex.printStackTrace();
        }
    }

    private void shutdown() {
        try {
            executor.shutdown();
            executor.awaitTermination(1, TimeUnit.HOURS);
        } catch (InterruptedException e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
