package ru.yandex.chemodan.videostreaming.framework.hls;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.SequenceInputStream;

import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.misc.ExceptionUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class AudioSegmentId3HeaderUtil {
    private static final String SIGNATURE = "ID3";

    private static final byte[] VERSION = {4, 0};

    private static final String PRIV_TAG_ID = "PRIV";

    private static final String PRIV_TAG_OWNER = "com.apple.streaming.transportStreamTimestamp";

    private static final int MPEG2_TS_TIME_BASE = 90000;

    public static InputStream prependTo(InputStream in, MediaTime timestamp) {
        return new SequenceInputStream(asInputStream(timestamp), in);
    }

    public static InputStream asInputStream(MediaTime timestamp) {
        return new ByteArrayInputStream(asByteArray(timestamp));
    }

    public static byte[] asByteArray(MediaTime timestamp) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        write(timestamp, baos);
        return baos.toByteArray();
    }

    public static void write(MediaTime timestamp, OutputStream out) {
        try {
            writeRaw(timestamp, out);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static void writeRaw(MediaTime timestamp, OutputStream out) throws IOException {
        DataOutputStream dataOut = new DataOutputStream(out);
        dataOut.writeBytes(SIGNATURE);
        dataOut.write(VERSION);
        dataOut.writeByte(0); // flags

        dataOut.writeInt(toSynchsafe(63)); // payload size

        dataOut.writeBytes(PRIV_TAG_ID);
        dataOut.writeInt(53); // tag payload size
        dataOut.writeShort(0); // flags

        dataOut.writeBytes(PRIV_TAG_OWNER);
        dataOut.writeByte(0); // NUL-terminator
        dataOut.writeLong(timestamp.getMicros() * MPEG2_TS_TIME_BASE / MediaTime.MICROS_PER_SECOND);
    }

    static int toSynchsafe(int value) {
        return (value & 0x7F)
                | ((value << 1) & 0x7F00)
                | ((value << 2) & 0x7F0000)
                | ((value << 3) & 0x7F000000);
    }
}
