package ru.yandex.chemodan.videostreaming.framework.hls;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class BitrateUtils {
    private static final Logger logger = LoggerFactory.getLogger(BitrateUtils.class);

    private enum BitrateThreshold {
        LOW_DIM(new Dimension(426, 240), 384),
        MIN_HQ(new Dimension(600, 400), 1152, LOW_DIM),
        MAX_HQ(new Dimension(1280, 720), 2560 * 2, MIN_HQ);

        final Option<BitrateThreshold> prevO;

        final Dimension dimension;

        final BitRate bitrate;

        BitrateThreshold(Dimension dimension, int bitrate) {
            this(dimension, bitrate, Option.empty());
        }

        BitrateThreshold(Dimension dimension, int bitrate, BitrateThreshold prev) {
            this(dimension, bitrate, Option.of(prev));
        }

        BitrateThreshold(Dimension dimension, int bitrate, Option<BitrateThreshold> prevO) {
            this.bitrate = BitRate.fromKbps(bitrate);
            this.dimension = dimension;
            this.prevO = prevO;
        }

        static BitrateThreshold byDimension(Dimension dimension) {
            for(BitrateThreshold threshold : BitrateThreshold.values()) {
                if (dimension.getSquare() < threshold.dimension.getSquare()) {
                    return threshold;
                }
            }
            return BitrateThreshold.MAX_HQ;
        }

        public BitRate getBitrate(Dimension dimension) {
            return prevO.map(p -> getBitrate(dimension, p))
                    .getOrElse(bitrate);
        }

        private BitRate getBitrate(Dimension dimension, BitrateThreshold prev) {
            long minSquare = prev.dimension.getSquare();
            long minBitrate = prev.bitrate.getKbps();
            long maxSquare = this.dimension.getSquare();
            long maxBitrate = this.bitrate.getKbps();

            double fraction = Math.max(0, 1.0 * dimension.getSquare() - minSquare) / (maxSquare - minSquare);
            int videoKbps = (int) (minBitrate + (maxBitrate - minBitrate) * fraction);
            logger.info("Video bit rate for {} : {}k", dimension, videoKbps);
            return BitRate.fromKbps(videoKbps);
        }
    }

    public static BitRate videoBitRateForDimension(Dimension dimension) {
        return BitrateThreshold.byDimension(dimension)
                .getBitrate(dimension);
    }

    private BitrateUtils() {}
}
